/*
 * RobodocDemo
 *
 * Author : Min Yang Jung, Peter Kazanzides
 * Created: February 18, 2013
 *
 * This application includes two components:
 *
 *  a) A robot simulator (either GLUT or OSG based)
 *  b) A test component to generate some simple robot motions
 *
 * This program contains the code to connect from the test component to the
 * ControlTask (in the RobotServer) and from the ControlTask to the simulator.
 * The test component does not require any Robodoc-specific code and it depends
 * only on the open source cisst/SAW library. 
 *
 * References:
 *    cisst: http://cisst.org/cisst
 *    SAW  : http://cisst.org/saw
 */

#include <cisstCommon.h>
#include <cisstMultiTask.h>
#include <cisstOSAbstraction/osaSleep.h>

#include <sawGLUTSimulator/osaGLUT.h>
#include <sawGLUTSimulator/mtsGLUTManipulator.h>

#include "config.h"
#include "TestComponent.h"
///////////////////////////////////////////////////////


int main(int argc, char **argv)
{
	/*vctMatRot3 aRot(-0.9786, 0.2041, 0.0263,
		-0.0234, -0.0159, -0.996,
		-0.2044, -0.9788, 0.0104, VCT_NORMALIZE);
	vctDouble3 aTrans(11.6578,-431.4891,116.0159);
	vctFrm3 atrns(aRot,aTrans);
	std::cout<<atrns<<std::endl;
	return 1;*/

	
    cmnLogger::SetMask(CMN_LOG_ALLOW_ALL);
    cmnLogger::SetMaskFunction(CMN_LOG_ALLOW_ALL);
    cmnLogger::AddChannel(std::cout, CMN_LOG_ALLOW_ERRORS_AND_WARNINGS);
    //cmnLogger::AddChannel(std::cout, CMN_LOG_ALLOW_ALL);
    cmnLogger::SetMaskDefaultLog(CMN_LOG_ALLOW_ALL);

    // get global component manager's IP from command line argument (default to localhost)
    std::string globalComponentManagerIP((argc < 2)?"localhost":argv[1]);
    std::cout << "Global component manager IP: " << globalComponentManagerIP << std::endl;

    // get local component manager
    const std::string thisProcessName("RobodocDemo2");
    mtsManagerLocal * LCM;
    try {
        LCM = mtsManagerLocal::GetInstance(globalComponentManagerIP, thisProcessName);
    } catch (...) {
        CMN_LOG_INIT_ERROR << "Failed to initialize local component manager" << std::endl;
        return 1;
    }

    // Setup GLUT
    osaGLUT glut(argc, argv);
    glut.SetSleepPeriod(10 * cmn_ms);

    // Setup robot model data file names
    std::string path(ROBODOC_DEMO_SOURCE_ROOT"/models/");
    std::vector<std::string> links;
    links.push_back(path + "l1.obj");
    links.push_back(path + "l2.obj");
    links.push_back(path + "l3.obj");
    links.push_back(path + "l4.obj");
    links.push_back(path + "l5_hip.obj");

    // Offset of the base
    vctFrame4x4<double> Rtb0(vctMatrixRotation3<double>(), 
                             vctFixedSizeVector<double,3>(0, 0, 0.3040));

    mtsGLUTManipulator manipulator("RobotSimulator", 
                               40 * cmn_ms,
                               OSA_CPUANY,
                               0,
                               links,
                               Rtb0,
                               path + "robodocdh.rob",
                               vctDoubleVec(5, 0.0),
                               path + "l0.obj");
    LCM->AddComponent(&manipulator);

    // Create test component
    TestComponent * robot = new TestComponent("Robodoc");
    LCM->AddComponent(robot);

    // connection for robot simulation
    CONNECT_REMOTE("ProcessCONTROL", "CONTROL", "RobotSim", 
                   thisProcessName, manipulator.GetName(), "Input");
    // connection for robot control
    CONNECT_REMOTE(thisProcessName, robot->GetName(), "Robot",
                   "ProcessCONTROL", "CONTROL", "prmRobot");

    // create the tasks, i.e. find the commands
    LCM->CreateAll();
    LCM->WaitForStateAll(mtsComponentState::READY);
    //osaSleep(1.0);

    //robodoc.Start();
    LCM->StartAll();
    LCM->WaitForStateAll(mtsComponentState::ACTIVE);

    // blocking call
    osaGLUT::StartMainLoop();

    // Cleanup and exit
    LCM->KillAll();
    LCM->Cleanup();
///////////////////////////////////////////////////////////////////
//		//free up all resources taken
//	Collection_Free(IdentifiedMarkers);
//	Xform3D_Free(PoseXf);
//	Cameras_Detach(); //important - otherwise the cameras will continue capturing, locking up this process.
///////////////////////////////////////////////////////////////////
    return 0;
}
