%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% An example script for connecting to the TexoCOM server, programming a
% sequence, running it, stopping it, and reading the cine buffer.
%
% Copyright: Ultrasonix Medical Corporation Jan 2013
% Author: Ali Baghani, Research Scientist, ali.baghani@ultrasonix.com
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Clearing the environment
clc
clear
close all

% Making the plot functions accessible to the environment
addpath('..\SonixDataTools\PlotFunctions');
addpath('..\SonixDataTools\ImageProcessing');
addpath('..\SonixDataTools\Misc');

% SDK path to find the TexoCOM server
SDK_BIN_PATH = '../../bin';                                                % YOU NEED TO CHANGE THIS LINE IF YOU MOVE THE CODE

% Registering the TexoCOM server on the computer and getting a handle to it
system(['cd "' SDK_BIN_PATH '" && ' 'regsvr32 TexoCOM.dll']);
h = actxserver('TexoCOM.Server');

% Choosing the connector to which the probe is connected
ConnectorWhereTheProbeIsConnected = 0;                                     % YOU NEED TO CHANGE THIS LINE TO REFLECT THE EXPERIMENTAL SETUP

% Some imaging parameters
gain  = 0.8;
power = 10;
depth = 100e3;

%% Initializing Texo
disp('Initializing Texo: programming the firmware');
FirmwarePath = '../../texo/dat/';                                          % YOU NEED TO CHANGE THIS LINE IF YOU MOVE THE CODE
success = h.texoInit(FirmwarePath, 3, 3, 0, 64, 0 , 128, true);            % YOU NEED TO CHANGE THIS LINE TO REFLECT YOUR HARDWARE
if(success)
    disp('Texo was successfully initialized!');
else
    disp('Texo could not be initialized!');
end

% Activating a probe connector
success = h.texoActivateProbeConnector(ConnectorWhereTheProbeIsConnected);
if(success)
    disp('The probe connector was successfully activated!');
else
    disp('Could not activate the connector!');
end

% Selecting the probe
h.texoSelectProbe(h.texoGetProbeCode(ConnectorWhereTheProbeIsConnected));
if(success)
    disp('The probe was successfully selected!');
else
    disp('Could not select the probe!');
end

%% Programing a sequence

% setting up a fixed tgc
h.texoClearTGCs;
h.texoAddTGCFixed(gain);

% setting up the power
h.texoSetPower(power, power, power);

% setting up the VCA
h.texoVCAInfo_setAmplification(16);
h.texoVCAInfo_setActivetermination(1);
h.texoVCAInfo_setInclamp(1600);
h.texoVCAInfo_setLPF(1);
h.texoVCAInfo_setLNAIntegratorEnable(1);
h.texoVCAInfo_setPGAIntegratorEnable(1);
h.texoVCAInfo_setHPFDigitalEnable(1);
h.texoVCAInfo_setHPFDigitalValue(11);
h.texoSetVCAInfo();

% the sequence
success = h.texoBeginSequence();
if(success)
    disp('Programming the sequence started!');
else
    disp('Failed to start programming the sequence!');
end

% setting transmit parameters
h.texoTP_setCenterElement(0);
h.texoTP_setAperture(64);
h.texoTP_setFocusDistance(depth * 1000 / 2);    % half the imaging depth
h.texoTP_setAngle(0);
h.texoTP_setFrequency(h.texoGetProbeCenterFreq());
h.texoTP_setPulseShape('+-');
h.texoTP_setSpeedOfSound(1540);
h.texoTP_setSync(0);
h.texoTP_setUseManualDelays(0);
h.texoTP_setUseMask(0);
h.texoTP_setTableIndex(-1);
h.texoTP_setTxDelay(100);
h.texoTP_setTxRepeat(0);
% setting receive parameters
h.texoRP_setCenterElement(0);
h.texoRP_setAperture(64);
h.texoRP_setAngle(0);
h.texoRP_setMaxApertureDepth(30000);
h.texoRP_setAcquisitionDepth(150e3);
h.texoRP_setSaveDelay(1 * 1000);
h.texoRP_setSpeedOfSound(1540);
h.texoRP_setChannelMask(true(1, 64));
h.texoRP_setApplyFocus(1);
h.texoRP_setUseManualDelays(0);
h.texoRP_setUseCustomWindow(0);
h.texoRP_setDecimation(0);
h.texoRP_setLGCValue(0);
h.texoRP_setTGCSel(0);
h.texoRP_setTableIndex(-1);
h.texoRP_setCustomLineDuration(0);
h.texoRP_setNumChannels(64);
h.texoRP_setWeightType(0);  % apodization

for Cntr = 0:127
    h.texoTP_setCenterElement(Cntr+0.5);
    h.texoRP_setCenterElement(Cntr+0.5);
    [success, lineSize, lineDuration] = h.texoAddLine;
    if (success)
        disp(['Successfully programmed line ' num2str(Cntr+0.5) ...
                 ' and the lineSize is ' num2str(lineSize) ...
                 ' and duration ' num2str(lineDuration)]);
    end
end

success = h.texoEndSequence();
if(success)
    disp('Programming the sequence finished!');
else
    disp('Failed to finish programming the sequence!');
end

%% Acquiring data
success = h.texoRunImage;
if(success)
    disp('Imaging started!');
else
    disp('Imaging could not be started');
end

% Wait for 1 seconds to acquire a few frames
pause(1); 

success = h.texoStopImage;
if(success)
    disp('Imaging stopped!');
else
    disp('Imaging could not be stopped');
end

disp(['Texo acquired ' h.texoGetCollectedFrameCount ' frames of RF data which are available in the cine']);

%% Reading the data into the Matlab environment amd displaying
data = h.texoGetCine;
data = reshape(data, [lineSize/2 128 h.texoGetCollectedFrameCount]);
% defining a header
header.filetype = 16;  % RF data
header.nframes  = size(data, 3);
header.w        = size(data, 2);
header.h        = size(data, 1);
header.ss       = 16;  % int16 data type for RF data
header.ul       = [1 1];
header.ur       = [size(data, 2) 1];
header.br       = [size(data, 2) size(data, 1)];
header.bl       = [1 size(data, 1)];
header.probe    = 0;
header.txf      = 0;
header.sf       = 0;
header.dr       = 0;
header.ld       = 0;
header.extra    = 0;
% Plotting
figure(1); hA = axes;
for Cntr = 1:size(data, 3)
    plot_SonixRP(data(:,:,Cntr), header, hA, 0.3);
    title(num2str(Cntr));
    pause(0.1)
end

%% Shutting down Texo
h.texoShutdown;
delete(h);
system(['cd "' SDK_BIN_PATH '" && ' 'regsvr32 /u TexoCOM.dll']);

