#include "StdAfx.h"
#include "header.h"

// default values for scan conversion (L14-5)
#define DEFAULT_WIDTH    640
#define DEFAULT_HEIGHT   480
#define DEFAULT_SCALE    80
#define DEFAULT_ORIGINX  320
#define DEFAULT_ORIGINY  10
#define DEFAULT_ELEMS    128
#define DEFAULT_PRBPITCH 300
#define DEFAULT_RADIUS   0
#define DEFAULT_TXOFFSET 0
#define DEFAULT_MAXSTEER 15000

#define swap(a, b) { temp = (a); (a) = (b); (b) = temp; }

// prototypes
void convertImage(const unsigned char* data, int size, unsigned int* cdata);
void invertImage(unsigned int* pData, int w, int h);
bool writeBitmap(const char* path, unsigned char* data, int w, int h, int bitdepth);

int rfToB(const char* in, const char* out);
int filterB(const char* in, const char* out);
int scanConvertB(const char* in, const char* out, const char* bmp);

int power(int base, int exp)
{
    // initial value for the result is the base
    int i, result = base;

    if (exp == 0)
    {
        return 1;
    }

    // multiplies the base by itself exp number of times
    for (i = 1; i < exp; i++)
    {
        result = result * base;
    }

    return result;
}

int main(int argc, char* argv[])
{
    if (argc < 5)
    {
        printf("error: not enough input parameters provided:\n");
        printf("usage:\n");
        printf(": amplio_demo_console <input_rf_file> <output_prescan_file> <output_filter_file> <output_scanconv_file> <output_bitmap>\n");
        printf("example:\n");
        printf("amplio_demo_console D:\\data.rf D:\\prescan.bpr D:\\filtered.bpr D:\\scanconverted.b8 D:\\final.bmp\n");
        return -1;
    }

    // apply envelope detection and log compression to the raw .rf data to get prescan b .bpr
    printf("\n\n\n Applying filtering, envelope detection and log compression ...:\n\n");
    if (rfToB(argv[1], argv[2]) == 0)
    {
        // apply clarity filter to the prescan b image .bpr
        printf("\n\n\n Applying ASR filter to image ...:\n\n");
        if (filterB(argv[2], argv[3]) == 0)
        {
            // apply scan conversion to the b image .b8 image
            printf("\n\n\n Applying scan conversion...:\n\n");
            scanConvertB(argv[3], argv[4], argv[5]);
        }
    }

    return 0;
}

int rfToB(const char* in, const char* out)
{
    int i, j;

    FILE* fp = fopen(in, "rb");
    FILE* fpb = fopen(out, "wb");

    if (!fp || !fpb)
    {
        fprintf(stderr, "could not open the source file\n");

        fclose(fp);
        fclose(fpb);

        return -1;
    }

    // read the rf header
    uFileHeader hdr;
    fread(&hdr, sizeof(hdr), 1, fp);

    int numLines = hdr.w;
    int numSamples = hdr.h;
    int numFrames = hdr.frames;
    int frameTag = 0;

    // IQ demodulation parameters
    int startDemFreq = hdr.txf;
    int endDemFreq = hdr.txf - 1000000;
    int cutOffFreq = 2000000;
    int samplingFreq = hdr.sf;
    // initialize Log Compression
    int useCompression = 1;    // if set to zero no log compression will be applied
    int dyn = 70; // dynamic range in dB
    int reject = 30; // rejection in dB

    // initialize IQ demodulation
    if (!amplioInitRfToB(startDemFreq, endDemFreq, cutOffFreq, samplingFreq, numSamples, useCompression, dyn, reject))  // load the demodulation and compression tables
    {
        fprintf(stderr, "could not initialize amplio\n");
        return -1;
    }

    // rf to bpr conversion parameters
    int decimation = 1; // 0: sampling freq, 1:sampling freq/2, 2:sampling freq/4, ...

    // modify the header for the bpr and write into the file
    int decimFactor = power(2, decimation);
    hdr.type = udtBPre;   // p-pre
    hdr.h = hdr.h / decimFactor;
    hdr.ss = 8;     // unsigned char
    hdr.sf = hdr.sf / decimFactor;
    fwrite(&hdr, sizeof(hdr), 1, fpb);

    // 1-read rf data line by line,
    // 2-process them to create bpr line,
    // 3-write into the file.
    short* rf = new short[numSamples];
    memset(rf, 0, numSamples * sizeof(short));
    unsigned char* b8 = new unsigned char[numSamples / decimFactor];
    memset(b8, 0, numSamples / decimFactor * sizeof(unsigned char));

    // for all the frames
    for (j = 0; j < numFrames; j++)
    {
        // read/write frame tag
        // Note: comment these two steps for Exam ver 5.x and later
        //fread(&frameTag, sizeof(int), 1, fp);
        //fwrite(&frameTag, sizeof(int), 1, fpb);

        // display the progress
        fprintf(stdout, "processing rf frame# %d ... frame Tag = %d ... [%d %%]\n", j, frameTag, (100 * (j + 1) / numFrames));

        // process each line separately
        for (i = 0; i < numLines; i++)
        {
            // read one RF line
            fread(rf, sizeof(short), numSamples, fp);
            // create envelope detected and log compressed b-line
            //m_amp.applyEnvelope(rf, numSamples, sumAfterEnvelope, useCompression, decimation, b8);
            amplioProcessRfToB(rf, decimation, b8);
            // write b-line
            fwrite(b8, sizeof(unsigned char), numSamples / decimFactor, fpb);
        }
    }

    fclose(fp);
    fclose(fpb);

    fprintf(stdout, "successfully stored .bpr data\n");

    delete[] rf;
    delete[] b8;

    return 0;
}

int filterB(const char* in, const char* out)
{
    FILE* fp;
    uFileHeader hdr;

    // default ASR settings
    int contrast = 20;
    int edge = 16;
    int lpcutoff = 10;
    int smooth = 13;
    int weight = 8;

    int sz;
    unsigned char* data = 0, * fdata = 0;

    // try to open input file
    fp = fopen(in, "rb");
    if (!fp)
    {
        fprintf(stderr, "could not open specified file\n");
        return -1;
    }

    // read file header
    fread(&hdr, sizeof(hdr), 1, fp);

    // ensure file type is 8 bit data
    // in this case, we have a prescan data
    // if scan conversion is applied, then we have udtBPost
    if (hdr.type != udtBPre && hdr.type != udtBPost)
    {
        fprintf(stderr, "invalid file type\n");
        fclose(fp);
        return -1;
    }

    // read frame header, does not apply to data acquire after version 5.x
    //if (hdr.type == udtBPre)
    //{
    //    fread(&frhdr, sizeof(int), 1, fp);
    //}

    // print data information
    printf("image information:\n");
    printf("- width: %d\n", hdr.w);
    printf("- height: %d\n", hdr.h);
    printf("\n");

    // read the data and apply the filter on each image frame
    sz = hdr.w * hdr.h;
    data = new unsigned char[sz];
    fdata = new unsigned char[sz * hdr.frames];

    // save the image as a .bpr file
    FILE* fpb = fopen(out, "wb");
    if (!fpb)
    {
        fprintf(stderr, "could not open output file for writing\n");
        return -1;
    }

    // write data header
    fwrite(&hdr, sizeof(hdr), 1, fpb);

    for (int j = 0; j < hdr.frames; j++)
    {
        // display the progress
        fprintf(stdout, "processing .bpr frame# %d ... [%d %%]\n", j, (100 * (j + 1) / hdr.frames));

        fread(data, sz, 1, fp);
        if (!amplioEnhance(data, fdata, (hdr.type == udtBPre) ? hdr.h : hdr.w, (hdr.type == udtBPre) ? hdr.w : hdr.h, edge, smooth, weight,
                contrast, lpcutoff))
        {
            fprintf(stderr, "could not perform filtering\n");
            return -1;
        }
        // write the filtered b image
        fwrite(fdata, sz, 1, fpb);
    }

    fclose(fp);
    fclose(fpb);

    printf("filtering was successful\n");

    delete[] data;
    delete[] fdata;

    return 0;
}

int scanConvertB(const char* in, const char* out, const char* bmp)
{
    FILE* fp;
    FILE* fp2;
    uFileHeader hdr;

    int sz, sz2;
    //int frhdr;
    int answer;
    unsigned char* data = 0, * image = 0;

    // try to open input file
    fp = fopen(in, "rb");
    if (!fp)
    {
        fprintf(stderr, "could not open specified file\n");
        return -1;
    }

    // read file header
    fread(&hdr, sizeof(hdr), 1, fp);

    // ensure file type is pre-scan data
    if (hdr.type != udtBPre)
    {
        fprintf(stderr, "invalid file type\n");
        fclose(fp);
        return -1;
    }

    // read frame header, Note does not apply to latest files
    //fread(&frhdr, sizeof(int), 1, fp);

    // print raw data information
    printf("pre-scan data information:\n");
    printf("- scanlines: %d\n", hdr.w);
    printf("- samples per line: %d\n", hdr.h);
    printf("- line density: %d\n", hdr.ld);
    printf("- sampling frequency: %d Hz\n", hdr.sf);
    printf("- collected from probe ID: %d\n", hdr.probe);
    printf("\n");

    // print default probe information
    int elems = DEFAULT_ELEMS, pitch = DEFAULT_PRBPITCH, radius = DEFAULT_RADIUS, maxsteer = DEFAULT_MAXSTEER, apexoffset =
        DEFAULT_TXOFFSET;
    // print default probe information
    printf("default output settings:\n");
    printf("- number of elements: %d \n", elems);
    printf("- pitch: %d microns\n", pitch);
    printf("- radius: %d microns\n", radius);
    printf("- max steering angle: %d degx1000\n", maxsteer);
    printf("- transmit offset: %d \n", apexoffset);
    printf("\n");
    // ask user if they want to change settings
    printf("would you like to change the default probe settings [1=yes,0=no] ");
    scanf("%d", &answer);

    if (answer == 1)
    {
        // obtain probe information
        printf("please provide probe information (see probes.xml):\n");
        printf("enter probe element count: ");
        scanf("%d", &elems);
        printf("enter probe pitch: ");
        scanf("%d", &pitch);
        printf("enter probe radius: ");
        scanf("%d", &radius);
        printf("enter probe max steer angle: ");
        scanf("%d", &maxsteer);
        printf("enter transmit offset: ");
        scanf("%d", &apexoffset);
    }

    int w = DEFAULT_WIDTH, h = DEFAULT_HEIGHT, scale = DEFAULT_SCALE, originy = DEFAULT_ORIGINY, originx = DEFAULT_ORIGINX; //w/2
    // print default output settings
    printf("default output settings:\n");
    printf("- output width: %d pixels\n", w);
    printf("- output height: %d pixels\n", h);
    printf("- scale: %d microns-per-pixel\n", scale);
    printf("- origin(x): %d pixels\n", originx);
    printf("- origin(y): %d pixels\n", originy);
    printf("\n");

    // ask user if they want to change settings
    printf("would you like to change the default output settings [1=yes,0=no] ");
    scanf("%d", &answer);

    if (answer == 1)
    {
        // obtain additional information
        printf("enter image output width: ");
        scanf("%d", &w);
        printf("enter image output height: ");
        scanf("%d", &h);
        printf("enter scaling in microns per pixel: ");
        scanf("%d", &scale);
        printf("enter originx: ");
        scanf("%d", &originx);
        printf("enter originy: ");
        scanf("%d", &originy);
    }

    // initialize the scan converter
    if (!amplioInitSC(w, h, hdr.h, hdr.w, 0, hdr.h, (hdr.ld - hdr.w) / 2, ((hdr.ld - hdr.w) / 2) + hdr.w, hdr.sf, hdr.ld,
            0, 0, originx, originy, scale, scale, elems, pitch, radius, maxsteer, apexoffset))
    {
        fprintf(stderr, "could not initialize scan converter\n");
        return -1;
    }

    // read pre-scan data
    sz = hdr.w * hdr.h;
    data = new unsigned char[sz];
    // try to scan convert
    sz2 = w * h;
    image = new unsigned char[sz2];

    fread(data, sz, 1, fp);
    fclose(fp);

    if (!amplioScanConvert(data, image))
    {
        fprintf(stderr, "could not perform scan conversion\n");
        return -1;
    }

    printf("scan conversion was successful\n");

    // save the scan-converted image if output file was provided as an argument
    if (strstr(bmp, ".bmp") != 0)
    {
        if (!writeBitmap(bmp, image, w, h, hdr.ss))
        {
            fprintf(stderr, "could not write bitmap\n");
            return -1;
        }
    }

    fp = fopen(in, "rb");
    fp2 = fopen(out, "wb+");

    if (!fp2)
    {
        fprintf(stderr, "could not open output file for writing\n");
        return -1;
    }

    // read file header
    fread(&hdr, sizeof(hdr), 1, fp);
    // modify the header for the bpr and write into the file
    hdr.type = udtBPost;   // p-post
    hdr.h = h;
    hdr.w = w;
    // write modified header
    fwrite(&hdr, sizeof(hdr), 1, fp2);

    for (int j = 0; j < hdr.frames; j++)
    {
        // display the progress
        fprintf(stdout, "processing b frame# %d ... [%d %%]\n", j, (100 * (j + 1) / hdr.frames));
        fread(data, sz, 1, fp);
        amplioScanConvert(data, image);
        fwrite(image, sz2, 1, fp2);
    }

    fclose(fp);
    fclose(fp2);

    printf("scan converted image storage was successful\n");

    delete[] data;
    delete[] image;

    return 0;
}

// converts 8 bit data into 32 bit RGB data
void convertImage(const unsigned char* data, int size, unsigned int* cdata)
{
    int i;
    for (i = 0; i < size; i++)
    {
        cdata[i] = data[i] << 16 | data[i] << 8 | data[i];
    }
}

/// inverts image data, so that the bitmap can be displayed properly
void invertImage(unsigned int* pData, int w, int h)
{
    int x, y, temp;
    unsigned int* p = (unsigned int*)pData;

    // flip the image on the X-axis
    for (y = 0; y < h / 2; y++)
    {
        for (x = 0; x < w; x++)
        {
            swap(p[y * w + x], p[((h - 1 - y) * w) + x]);
        }
    }
}

/// write a bitmap file to a specified location
bool writeBitmap(const char* path, unsigned char* data, int w, int h, int ss)
{
    FILE* fp = fopen(path, "wb");
    if (!fp)
    {
        return false;
    }

    unsigned int* cdata;

    if (ss == 32)
    {
        cdata = (unsigned int*)data;
    }
    else
    {
        cdata = new unsigned int[w * h];
    }

    // convert 8 bit data to 32 bit data
    convertImage(data, w * h, cdata);
    // since the data gets flipped when written to bitmap, pre-flip the image
    invertImage((unsigned int*)cdata, w, h);

    BITMAPINFOHEADER BMIH;
    BMIH.biSize = sizeof(BITMAPINFOHEADER);
    BMIH.biBitCount = (unsigned short)32;
    BMIH.biPlanes = 1;
    BMIH.biCompression = BI_RGB;
    BMIH.biWidth = w;
    BMIH.biHeight = h;
    BMIH.biSizeImage = ((((BMIH.biWidth * BMIH.biBitCount) + 31) & ~31) >> 3) * BMIH.biHeight;

    BITMAPFILEHEADER bmfh;
    int nBitsOffset = sizeof(BITMAPFILEHEADER) + BMIH.biSize;
    LONG lImageSize = BMIH.biSizeImage;
    LONG lFileSize = nBitsOffset + lImageSize;
    bmfh.bfType = 'B' + ('M' << 8);
    bmfh.bfOffBits = nBitsOffset;
    bmfh.bfSize = lFileSize;
    bmfh.bfReserved1 = bmfh.bfReserved2 = 0;
    // write the bitmap file header
    fwrite(&bmfh, 1, sizeof(BITMAPFILEHEADER), fp);
    // and then the bitmap info header
    fwrite(&BMIH, 1, sizeof(BITMAPINFOHEADER), fp);
    // finally, write the image data itself
    fwrite(cdata, 1, lImageSize, fp);

    fclose(fp);

    if (ss == 32)
    {
        delete[] cdata;
    }

    return true;
}
