#include "stdafx.h"
#include "AmplioDemo.h"
#include "header.h"

#define MSG_TIMEOUT 3000     // ms

// default values for scan conversion (L14-5)

// default values for scan conversion (L14-5)

AmplioDemo::AmplioDemo(QWidget* parent) : QMainWindow(parent)
{
    setupUi(this);

    m_cineSize = 0;
    m_numLines = 0;
    m_numRFSamples = 0;
    m_numBSamples = 0;
    m_numFrames = 0;

    // buffers
    m_rf = 0;
    m_b = 0;
    m_bflt = 0;
    m_bsc = 0;

    wProgress = new QProgressBar();
    wProgress->setMaximumSize(350, QWIDGETSIZE_MAX);
    wTool->addWidget(wProgress);
    wProgress->setRange(0, 100);
    wProgress->setValue(0);

    m_isInitialized = false;
}

AmplioDemo::~AmplioDemo()
{
    if (m_rf)
    {
        delete[] m_rf;
        m_rf = 0;
    }

    if (m_b)
    {
        delete[] m_b;
        m_b = 0;
    }

    if (m_bflt)
    {
        delete[] m_bflt;
        m_bflt = 0;
    }

    if (m_bsc)
    {
        delete[] m_bsc;
        m_bsc = 0;
    }
}

void AmplioDemo::cineScroll(int index)
{
    // update RF display
    if (index < 0 || index > (m_numFrames - 1))
    {
        return;
    }

    FILE* fp = fopen(m_inRFPath, "rb");

    if (!fp || !m_rf)
    {
        fprintf(stderr, "could not open the source file\n");
    }
    else
    {
        // read the rf frame of interest
        int sz = m_numLines * m_numRFSamples * sizeof(short);
        fseek(fp, sizeof(uFileHeader) + index * sz, SEEK_SET);
        fread(m_rf, sz, 1, fp);
        fclose(fp);
        // display the image
        rfDisplay->setImgData((char*)m_rf);
    }

    // update original B
    fp = fopen(m_outBprPath, "rb");
    if (!fp || !m_b)
    {
        fprintf(stderr, "could not open the source file\n");
    }
    else
    {
        // read the rf frame of interest
        int sz = m_numLines * m_numBSamples;
        fseek(fp, sizeof(uFileHeader) + index * sz, SEEK_SET);
        fread(m_b, sz, 1, fp);
        fclose(fp);
        // display the image
        bDisplay->setImgData((char*)m_b);
    }

    // update ASR filtered B
    fp = fopen(m_outBprFiltPath, "rb");
    if (!fp || !m_bflt)
    {
        fprintf(stderr, "could not open the source file\n");
    }
    else
    {
        // read the rf frame of interest
        int sz = m_numLines * m_numBSamples;
        fseek(fp, sizeof(uFileHeader) + index * sz, SEEK_SET);
        fread(m_bflt, sz, 1, fp);
        fclose(fp);
        // display the image
        bfDisplay->setImgData((char*)m_bflt);
    }

    // update scan converted B
    fp = fopen(m_outBpostPath, "rb");
    if (!fp || !m_bsc)
    {
        fprintf(stderr, "could not open the source file\n");
    }
    else
    {
        // read the rf frame of interest
        int sz = m_sc.width * m_sc.height;
        fseek(fp, sizeof(uFileHeader) + index * sz, SEEK_SET);
        fread(m_bsc, sz, 1, fp);
        fclose(fp);
        // display the image
        bsDisplay->setImgData((char*)m_bsc);
    }
}

void AmplioDemo::onAmp(int value)
{
    rfDisplay->setAmp(value);
}

void AmplioDemo::onApplyParams()
{
    // quadrature demodulation params
    m_amplio.startFreq = wParams->item(0, 0)->text().toInt();
    m_amplio.stopFreq = wParams->item(1, 0)->text().toInt();
    m_amplio.cutoffFreq = wParams->item(2, 0)->text().toInt();
    // Compression table params
    m_amplio.reject = wParams->item(3, 0)->text().toInt();
    m_amplio.dynamicRg = wParams->item(4, 0)->text().toInt();
    m_amplio.decimation = wParams->item(5, 0)->text().toInt();
    // ASR params
    m_asr.contrast = wParams->item(6, 0)->text().toInt();
    m_asr.edge = wParams->item(7, 0)->text().toInt();
    m_asr.lpcutoff = wParams->item(8, 0)->text().toInt();
    m_asr.smooth = wParams->item(9, 0)->text().toInt();
    m_asr.weight = wParams->item(10, 0)->text().toInt();
    // probe params
    m_probe.elements = wParams->item(11, 0)->text().toInt();
    m_probe.pitch = wParams->item(12, 0)->text().toInt();
    m_probe.radius = wParams->item(13, 0)->text().toInt();
    m_probe.txoffset = wParams->item(14, 0)->text().toInt();
    m_probe.maxsteer = wParams->item(15, 0)->text().toInt();
    // scan conversion params
    m_sc.width = wParams->item(16, 0)->text().toInt();
    m_sc.height = wParams->item(17, 0)->text().toInt();
    m_sc.originx = wParams->item(18, 0)->text().toInt();
    m_sc.originy = wParams->item(19, 0)->text().toInt();
    m_sc.scale = wParams->item(20, 0)->text().toInt();
}

void AmplioDemo::onSelectFile()
{
    m_isInitialized = false;

    QString fileName = QFileDialog::getOpenFileName(this,
            tr("Open RF File"), "/", tr("Text Files (*.rf)"));
    fileName.chop(3);   // drop ".rf"
    QByteArray ba = fileName.toLocal8Bit();
    const char* path = ba.data();
    strcpy(m_inRFPath, path);
    strcpy(m_outBprPath, path);
    strcpy(m_outBprFiltPath, path);
    strcpy(m_outBpostPath, path);

    strcat(m_inRFPath, ".rf");
    strcat(m_outBprPath, "_qt.bpr");
    strcat(m_outBprFiltPath, "_qt_asr.bpr");
    strcat(m_outBpostPath, "_qt.b8");

    onApplyParams();

    m_isInitialized = true;
    wStatus->showMessage(tr("Successfully selected the RF file"));
}

void AmplioDemo::runImage()
{
    if (!m_isInitialized)
    {
        QMessageBox mb;
        mb.setText(tr("Rf file needs to be selected first!"));
        mb.exec();
        return;
    }

    // apply envelope detection and log compression to the raw .rf data to get prescan b .bpr
    printf("\n\n\n Applying filtering, envelope detection and log compression ...:\n\n");
    wStatus->showMessage(tr("Processing RF data ..."));
    if (!rfToB(m_inRFPath, m_outBprPath))
    {
        printf("RF processing failed!\n");
    }
    else
    {
        wStatus->showMessage(tr("Successfully Processed RF data"));
    }

    // init cine size
    wCine->setRange(0, m_numFrames - 1);
    wCine->setValue(wCine->maximum());

    // apply clarity filter to the prescan b image .bpr
    printf("\n\n\n Applying ASR filter to image ...:\n\n");
    wStatus->showMessage(tr("Applying Speckle Reduction on B data ..."));
    if (!filterB(m_outBprPath, m_outBprFiltPath))
    {
        printf("Applying Speckle Reduction failed!\n");
    }
    else
    {
        wStatus->showMessage(tr("Successfully Applied Speckle Reduction on B data"));
    }

    // apply scan conversion to the b image .b8 image
    printf("\n\n\n Applying scan conversion...:\n\n");
    wStatus->showMessage(tr("Applying Scan Conversion..."));
    if (!scanConvertB(m_outBprFiltPath, m_outBpostPath))
    {
        printf("Applying Scan Conversion failed!\n");
    }
    else
    {
        wStatus->showMessage(tr("Successfully Applied Scan Conversion"));
    }

    // reset memory buffers
    resetBuffers(m_numLines * m_numRFSamples, m_numLines * m_numBSamples, m_sc.width * m_sc.height);
}

bool AmplioDemo::rfToB(const char* in, const char* out)
{
    int i, j;

    FILE* fp = fopen(in, "rb");
    FILE* fpb = fopen(out, "wb");

    if (!fp || !fpb)
    {
        fprintf(stderr, "could not open the source file\n");
        return false;
    }

    // read the rf header
    uFileHeader hdr;
    fread(&hdr, sizeof(hdr), 1, fp);

    // print headerinformation
    printf("header information:\n");
    printf("- frames : %d\n", hdr.frames);
    printf("- scanlines: %d\n", hdr.w);
    printf("- samples per line: %d\n", hdr.h);
    printf("- line density: %d\n", hdr.ld);
    printf("- transmit frequency: %d Hz\n", hdr.txf);
    printf("- sampling frequency: %d Hz\n", hdr.sf);
    printf("- collected from probe ID: %d\n", hdr.probe);
    printf("\n");

    if (hdr.w <= 0 || hdr.h <= 0 || hdr.frames <= 0 || hdr.txf <= 0 || hdr.sf <= 0)
    {
        QMessageBox mb;
        mb.setText(tr("The header file contains invalid information!"));
        mb.exec();
        return false;
    }

    m_numLines = hdr.w;
    m_numRFSamples = hdr.h;
    m_numFrames = hdr.frames;

    // init Rf display
    rfDisplay->init(m_numLines, m_numRFSamples);

    // IQ demodulation parameters
    int samplingFreq = hdr.sf;
    int startDemFreq = m_amplio.startFreq;
    int endDemFreq = m_amplio.stopFreq;
    int cutOffFreq = m_amplio.cutoffFreq;
    // initialize Log Compression
    int useCompression = 1;    // if set to zero no log compression will be applied
    int reject = m_amplio.reject; // rejection in dB
    int dyn = m_amplio.dynamicRg; // dynamic range in dB

    // initialize IQ demodulation
    if (!amplioInitRfToB(startDemFreq, endDemFreq, cutOffFreq, samplingFreq, m_numRFSamples, useCompression, dyn, reject))  // load the demodulation and compression tables
    {
        fprintf(stderr, "could not initialize amplio\n");
        return false;
    }

    // rf to bpr conversion parameters
    int decimation = m_amplio.decimation; // 0: sampling freq, 1:sampling freq/2, 2:sampling freq/4, ...
    int decimFactor = 1 << decimation;
    m_numBSamples = m_numRFSamples / decimFactor;

    // init B display
    bDisplay->init(m_numLines, m_numBSamples);

    // modify the header for the bpr and write into the file
    hdr.type = udtBPre;   // p-pre
    hdr.h = hdr.h / decimFactor;
    hdr.ss = 8;     // unsigned char
    hdr.sf = hdr.sf / decimFactor;
    fwrite(&hdr, sizeof(hdr), 1, fpb);

    // 1-read rf data line by line,
    // 2-process them to create bpr line,
    // 3-write into the file.
    allocateBuffers(m_numLines * m_numRFSamples, m_numLines * m_numBSamples, m_sc.width * m_sc.height);

    // for all the frames
    for (j = 0; j < m_numFrames; j++)
    {
        wProgress->setValue((100 * (j + 1) / m_numFrames));

        // read/write frame tag
        // Note: comment these two steps for Exam ver 5.x and later
        //fread(&frameTag, sizeof(int), 1, fp);
        //fwrite(&frameTag, sizeof(int), 1, fpb);

        // display the progress
        fprintf(stdout, "processing rf frame# %d ... [%d %%]\n", j, (100 * (j + 1) / m_numFrames));

        // process each line separately
        for (i = 0; i < m_numLines; i++)
        {
            // read one RF line
            short* rfline = &m_rf[i * m_numRFSamples];
            unsigned char* bline = &m_b[i * m_numBSamples];

            fread(rfline, sizeof(short), m_numRFSamples, fp);
            // create envelope detected and log compressed b-line
            //m_amp.applyEnvelope(rf, numSamples, sumAfterEnvelope, useCompression, decimation, b8);
            amplioProcessRfToB(rfline, decimation, bline);
            // write b-line
            fwrite(bline, sizeof(unsigned char), m_numBSamples, fpb);
        }
    }

    fclose(fp);
    fclose(fpb);

    fprintf(stdout, "successfully stored .bpr data\n");

    rfDisplay->setImgData((char*)m_rf);
    bDisplay->setImgData((char*)m_b);

    return true;
}

bool AmplioDemo::filterB(const char* in, const char* out)
{
    FILE* fp;
    uFileHeader hdr;

    // init display
    bfDisplay->init(m_numLines, m_numBSamples);

    // default ASR settings
    int contrast = m_asr.contrast;
    int edge = m_asr.edge;
    int lpcutoff = m_asr.lpcutoff;
    int smooth = m_asr.smooth;
    int weight = m_asr.weight;

    int sz = m_numLines * m_numBSamples;

    // try to open input file
    fp = fopen(in, "rb");
    if (!fp)
    {
        fprintf(stderr, "could not open specified file\n");
        return false;
    }

    // read file header
    fread(&hdr, sizeof(hdr), 1, fp);

    // ensure file type is 8 bit data
    // in this case, we have a prescan data
    // if scan conversion is applied, then we have udtBPost
    if (hdr.type != udtBPre && hdr.type != udtBPost)
    {
        fprintf(stderr, "invalid file type\n");
        fclose(fp);
        return false;
    }

    // read frame header, does not apply to data acquire after version 5.x
    //if (hdr.type == udtBPre)
    //{
    //    fread(&frhdr, sizeof(int), 1, fp);
    //}

    // print data information
    printf("image information:\n");
    printf("- width: %d\n", hdr.w);
    printf("- height: %d\n", hdr.h);
    printf("\n");

    // read the data and apply the filter on each image frame
    // save the image as a .bpr file
    FILE* fpb = fopen(out, "wb");
    if (!fpb)
    {
        fprintf(stderr, "could not open output file for writing\n");
        return false;
    }

    // write data header
    fwrite(&hdr, sizeof(hdr), 1, fpb);

    for (int j = 0; j < hdr.frames; j++)
    {
        // display the progress
        fprintf(stdout, "processing .bpr frame# %d ... [%d %%]\n", j, (100 * (j + 1) / hdr.frames));
        // progressBar
        wProgress->setValue((100 * (j + 1) / hdr.frames));

        fread(m_b, sz, 1, fp);
        if (!amplioEnhance(m_b, m_bflt, (hdr.type == udtBPre) ? hdr.h : hdr.w, (hdr.type == udtBPre) ? hdr.w : hdr.h,
                edge, smooth, weight, contrast, lpcutoff))
        {
            fprintf(stderr, "could not perform filtering\n");
            fclose(fp);
            fclose(fpb);
            return false;
        }
        // write the filtered b image
        fwrite(m_bflt, sz, 1, fpb);
    }

    fclose(fp);
    fclose(fpb);

    printf("filtering was successful\n");

    bfDisplay->setImgData((char*)m_bflt);

    return true;
}

bool AmplioDemo::scanConvertB(const char* in, const char* out)
{
    FILE* fp;
    FILE* fpb;
    uFileHeader hdr;

    int sz, sz2;

    // try to open input file
    fp = fopen(in, "rb");
    fpb = fopen(out, "wb");

    if (!fpb || !fp)
    {
        fprintf(stderr, "could not open specified file\n");
        return false;
    }

    // read file header
    fread(&hdr, sizeof(hdr), 1, fp);

    // ensure file type is pre-scan data
    if (hdr.type != udtBPre)
    {
        fprintf(stderr, "invalid file type\n");
        fclose(fp);
        fclose(fpb);
        return false;
    }

    // read frame header, Note does not apply to latest files
    //fread(&frhdr, sizeof(int), 1, fp);

    // print raw data information
    printf("pre-scan data information:\n");
    printf("- scanlines: %d\n", hdr.w);
    printf("- samples per line: %d\n", hdr.h);
    printf("- line density: %d\n", hdr.ld);
    printf("- sampling frequency: %d Hz\n", hdr.sf);
    printf("- collected from probe ID: %d\n", hdr.probe);
    printf("\n");

    // probe params
    int elems = m_probe.elements, pitch = m_probe.pitch, radius = m_probe.radius,
        maxsteer = m_probe.maxsteer, apexoffset = m_probe.txoffset;

    // scan conversion params
    int w = m_sc.width, h = m_sc.height, scale = m_sc.scale,
        originy = m_sc.originy, originx = m_sc.originx;

    // init display
    bsDisplay->init(h, w, true);

    // print default output settings
    printf("default output settings:\n");
    printf("- output width: %d pixels\n", w);
    printf("- output height: %d pixels\n", h);
    printf("- scale: %d microns-per-pixel\n", scale);
    printf("- origin(x): %d pixels\n", originx);
    printf("- origin(y): %d pixels\n", originy);
    printf("\n");

    // initialize the scan converter
    if (!amplioInitSC(w, h, hdr.h, hdr.w, 0, hdr.h, (hdr.ld - hdr.w) / 2, ((hdr.ld - hdr.w) / 2) + hdr.w, hdr.sf, hdr.ld,
            0, 0, originx, originy, scale, scale, elems, pitch, radius, maxsteer, apexoffset))
    {
        fprintf(stderr, "could not initialize scan converter\n");
        fclose(fp);
        fclose(fpb);
        return false;
    }

    //keep the size of pre-scan data
    sz = hdr.w * hdr.h;

    // modify the header for the bpr and write into the file
    hdr.type = udtBPost;    // p-post
    hdr.h = h;
    hdr.w = w;

    // write modified header
    fwrite(&hdr, sizeof(hdr), 1, fpb);

    // calculate the size of post scan data
    sz2 = hdr.h * hdr.w;

    for (int j = 0; j < hdr.frames; j++)
    {
        //allocateBuffers(m_numLines * m_numRFSamples, m_numLines * m_numBSamples, m_sc.width * m_sc.height);

        wProgress->setValue((100 * (j + 1) / hdr.frames));
        // display the progress
        fprintf(stdout, "processing b frame# %d ... [%d %%]\n", j, (100 * (j + 1) / hdr.frames));
        fread(m_bflt, sz, 1, fp);
        if (!amplioScanConvert(m_bflt, m_bsc))
        {
            fprintf(stderr, "scan conversion failed\n");
        }
        fwrite(m_bsc, sz2, 1, fpb);
    }

    fclose(fp);
    fclose(fpb);

    printf("scan converted image storage was successful\n");

    bsDisplay->setImgData((char*)m_bsc);

    return true;
}

void AmplioDemo::allocateBuffers(int szRF, int szB, int szB8)
{
    if (m_rf)
    {
        delete[] m_rf;
        m_rf = 0;
    }

    if (m_b)
    {
        delete[] m_b;
        m_b = 0;
    }

    if (m_bflt)
    {
        delete[] m_bflt;
        m_bflt = 0;
    }

    if (m_bsc)
    {
        delete[] m_bsc;
        m_bsc = 0;
    }

    m_rf = new short[szRF];
    m_b = new unsigned char[szB];
    m_bflt = new unsigned char[szB];
    m_bsc = new unsigned char[szB8];

    resetBuffers(szRF, szB, szB8);
}

void AmplioDemo::resetBuffers(int szRF, int szB, int szB8)
{
    if (m_rf)
    {
        memset(m_rf, 0, szRF * sizeof(short));
    }

    if (m_b)
    {
        memset(m_b, 0, szB);
    }

    if (m_bflt)
    {
        memset(m_bflt, 0, szB);
    }

    if (m_bsc)
    {
        memset(m_bsc, 0, szB8);
    }
}
