#include "stdafx.h"
#include "ViewBImg.h"
#include "math.h"

#define bound(y, h) (y < 0 ? ((-y > h) ? -h : y) : ((y > h) ? h : y))

ViewBImg::ViewBImg(QWidget* parent) : QGraphicsView(parent)
{
    QGraphicsScene* sc = new QGraphicsScene(this);
    setScene(sc);

    m_scanLines = 32;
    m_samples = 0;
    m_buffer = 0;
    m_bufferImg = 0;
    m_image = 0;
    m_amp = 100;
    m_map = 0;
    m_scanConverted = false;
}

ViewBImg::~ViewBImg()
{
    if (m_buffer)
    {
        delete[] m_buffer;
    }

    if (m_bufferImg)
    {
        delete[] m_bufferImg;
    }

    if (m_map)
    {
        delete[] m_map;
    }
}

bool ViewBImg::init(int scanLines, int samples, bool scanConverted)
{
    m_scanLines = scanLines;
    m_samples = samples;
    m_scanConverted = scanConverted;

    if (m_buffer)
    {
        delete[] m_buffer;
    }

    if (m_bufferImg)
    {
        delete[] m_bufferImg;
    }

    int sz = m_scanLines * m_samples;
    if (sz)
    {
        // RGB buffer for the image
        m_buffer = new char[sz];
        m_bufferImg = new unsigned char[sz * 4];
    }

    int w, h;
    getOptDims(w, h);
    setupBuffer(w, h);
    setSceneRect(0, 0, width(), height());

    if (m_map)
    {
        delete[] m_map;
    }
    // create the RGB colormap
    m_map = new int[256];
    for (int i = 0; i < 256; i++)
    {
        m_map[i] = (i + (i << 8) + (i << 16));
    }

    return true;
}

void ViewBImg::resizeEvent(QResizeEvent* e)
{
    setSceneRect(0, 0, e->size().width(), e->size().height());
    int w, h;
    getOptDims(w, h);
    setupBuffer(w, h);

    QGraphicsView::resizeEvent(e);
}

void ViewBImg::drawBackground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawBackground(painter, r);

    if (m_image)
    {
        QRect r1(0, 0, width(), height());
        int w, h;
        getOptDims(w, h);
        QRect r2(0, 0, w, h);
        painter->drawImage(r1, *m_image, r2, Qt::MonoOnly);
    }
    else
    {
        painter->fillRect(0, 0, width(), height(), Qt::black);
    }
}

void ViewBImg::convertRF2Img(char* buffer, unsigned char* bufferImg)
{
    unsigned char* data = (unsigned char*)(buffer);
    int* img = (int*)(bufferImg);

    for (int j = 0; j < m_scanLines; j++)
    {
        for (int i = 0; i < m_samples; i++)
        {
            int b = data[j * m_samples + i];
            int val = bound(b * m_amp / 100, 255);
            if (m_scanConverted)
            {
                img[j * m_samples + i] = m_map[(unsigned char)val];
            }
            else
            {
                //transpose and apply colormap
                img[i * m_scanLines + j] = m_map[(unsigned char)val];
            }
        }
    }
}

bool ViewBImg::setImgData(char* data)
{
    if (!m_buffer || !m_bufferImg || !m_image)
    {
        return false;
    }

    memcpy(m_buffer, data, m_scanLines * m_samples);
    convertRF2Img(m_buffer, m_bufferImg);

    scene()->invalidate();
    return true;
}

void ViewBImg::getOptDims(int& w, int& h)
{
    if (m_scanConverted)
    {
        w = m_samples;
        h = m_scanLines;
    }
    else
    {
        //transpose the image
        w = m_scanLines;
        h = m_samples;
    }
}

void ViewBImg::setupBuffer(int w, int h)
{
    if (m_image)
    {
        delete m_image;
    }

    m_image = new QImage(m_bufferImg, w, h, QImage::Format_RGB32);
}

void ViewBImg::setAmp(int amp)
{
    if (amp > 0)
    {
        m_amp = amp;
    }
    scene()->invalidate();
}
