#include "stdafx.h"
#include "ViewChannels.h"

#define bound(y, h) (y < 0 ? ((-y > h) ? -h : y) : ((y > h) ? h : y))

ViewChannels::ViewChannels(QWidget* parent) : QGraphicsView(parent)
{
    QGraphicsScene* sc = new QGraphicsScene(this);
    setScene(sc);

    m_scanLines = 0;
    m_display = 32;
    m_samples = 0;
    m_ssize = 2;
    m_scale = 1.0;
    m_amp = 0;
    m_buffer = 0;
}

ViewChannels::~ViewChannels()
{
    if (m_buffer)
    {
        delete[] m_buffer;
    }
}

bool ViewChannels::init(int scanLines, int samples)
{
    m_scanLines = scanLines;
    m_samples = samples;
    m_ssize = 2;

    if (m_buffer)
    {
        delete[] m_buffer;
    }

    if (m_samples)
    {
        m_buffer = new char[m_scanLines * m_samples * m_ssize];
        memset(m_buffer, 0, m_scanLines * m_samples * m_ssize);
    }

    adjustScale();
    return true;
}

void ViewChannels::adjustScale()
{
    if (m_samples)
    {
        m_scale = (double)width() / (double)m_samples;
    }
}

void ViewChannels::setAmp(int amp)
{
    m_amp = amp;
    scene()->invalidate();
}

void ViewChannels::setChannelDisplay(int chd)
{
    m_display = chd;
    scene()->invalidate();
}

void ViewChannels::display(char* data)
{
    if (m_buffer)
    {
        memcpy(m_buffer, data, m_scanLines * m_samples * m_ssize);
        scene()->invalidate();
    }
}

void ViewChannels::resizeEvent(QResizeEvent* e)
{
    setSceneRect(0, 0, e->size().width(), e->size().height());

    adjustScale();

    QGraphicsView::resizeEvent(e);
}

void ViewChannels::drawForeground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawForeground(painter, r);

    if (m_buffer)
    {
        int h = height() / m_display;
        double scale = ((double)m_amp / 100.0);
        short* sample = (short*)m_buffer;

        painter->setFont(QFont("Arial", h - 4));

        for (int i = 0; i < m_display; i++)
        {
            painter->scale(m_scale, 1);
            painter->setPen(Qt::red);
            QPainterPath pp(QPointF(0, ((i * h) + (h / 2)) + (sample[i * m_samples] * scale)));
            for (int x = 1; x < m_samples; x++)
            {
                auto y = bound((sample[i * m_samples + x] * scale), (h / 2));
                pp.lineTo(x, ((i * h) + (h / 2)) + y);
            }
            painter->drawPath(pp);

            painter->setPen(Qt::black);
            painter->drawRect(0, i * h, m_samples - 1, h - 1);

            painter->resetTransform();
            painter->drawText(1, ((i + 1) * h) - 2, QString("%1").arg(i + 1));
        }
    }
}
