#include "stdafx.h"
#include "AdvDlg.h"

AdvDlg::AdvDlg()
{
    setupUi(this);
    onRefresh();
    onChangeAction(0);
}

void AdvDlg::onClear()
{
    wOutput->clear();
}

void AdvDlg::onConnect()
{
    int index = wDevices->currentIndex();

    if (index < 0)
    {
        return;
    }

    QString name = wDevices->itemText(index);

    if (daqConnect(index))
    {
        wOutput->appendPlainText(QString(tr("Connected to %1")).arg(name));
    }
    else
    {
        wOutput->appendPlainText(QString(tr("Failed to connect to %1")).arg(name));
    }
}

void AdvDlg::onRefresh()
{
    wOutput->clear();
    wOutput->appendPlainText(tr("Querying for DAQ devices..."));

    char dev[256];
    int index = 0;
    std::vector< std::string > devices;

    while (daqGetDeviceList(index++, dev, 256))
    {
        devices.push_back(dev);
    }

    wOutput->appendPlainText(QString(tr("Found %1 device(s)")).arg(devices.size()));

    wDevices->clear();

    for (size_t i = 0; i < devices.size(); ++i)
    {
        wDevices->addItem(devices[i].c_str());
    }

    if (devices.size())
    {
        wDevices->setCurrentIndex(0);
    }
}

void AdvDlg::onTestDDR()
{
    QString pattern;
    wOutput->appendPlainText(tr("Running DDR2 test with pattern..."));

    switch (wDDRPattern->currentIndex())
    {
    case 1:
        pattern = "0xAAAAAAAA";
        break;
    case 2:
        pattern = "0x55555555";
        break;
    default:
        pattern = "Counter";
        break;
    }

    if (daqDoAdvanced(advDDR2Test, wDDRPattern->currentIndex()))
    {
        wOutput->appendPlainText(QString(tr("Test pattern '%1' was successful")).arg(pattern));
    }
    else
    {
        wOutput->appendPlainText(QString(tr("Test pattern '%1' failed")).arg(pattern));
    }
}

void AdvDlg::onUSB()
{
    wOutput->appendPlainText(tr("Checking USB connection..."));

    if (daqCheckUSB())
    {
        wOutput->appendPlainText(tr("USB controller is configured as high speed"));
    }
    else
    {
        wOutput->appendPlainText(tr("USB controller is NOT configured as high speed"));
    }
}

void AdvDlg::onMainTest()
{
    bool pass;
    int b[] = { wRxB1->isChecked() ? 1 : 0, wRxB2->isChecked() ? 1 : 0, wRxB3->isChecked() ? 1 : 0, wRxB4->isChecked() ? 1 : 0 };
    unsigned int testVal = (unsigned int)rand() % 255;

    EDeviceID devices[] = { eRX1Reg, eRX2Reg, eRX3Reg, eRX4Reg };

    wOutput->appendPlainText(tr("Programming RX Clock..."));
    if (daqDoAdvanced(advProgramRxClock, b[0], b[1], b[2], b[3]))
    {
        wOutput->appendPlainText(tr("Passed"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed"));
    }

    wOutput->appendPlainText(tr("Programming RX phase alignment..."));
    if (daqDoAdvanced(advProgramRxPhase, b[0], b[1], b[2], b[3]))
    {
        wOutput->appendPlainText(tr("Passed"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed"));
    }

    wOutput->appendPlainText(tr("Programming RX PC registers..."));

    unsigned int i, addr, readBack;
    for (i = 0; i < 4; ++i)
    {
        if (!b[i])
        {
            continue;
        }

        pass = true;

        for (addr = 0x00100; addr <= 0x00103; ++addr)
        {
            daqDoAdvanced(advWriteDevice, devices[i], addr, testVal);
            readBack = daqDoAdvanced(advReadDevice, devices[i], addr);
            if (!(readBack == testVal))
            {
                pass = false;
                break;
            }
        }

        if (pass)
        {
            wOutput->appendPlainText(QString(tr("Passed board %1")).arg(i + 1));
        }
        else
        {
            wOutput->appendPlainText(QString(tr("Failed board %1")).arg(i + 1));
        }
    }
}

void AdvDlg::onDeviceHealth()
{
    float temperature[DEVICE_HEALTH_TEMPS], voltage[DEVICE_HEALTH_VOLTS];
    int i, j = 0;
    QString msg;
    char err[256];

    wOutput->appendPlainText(tr("Attempting to query temperatures and voltages..."));

    if (!daqGetDeviceHealth(temperature, voltage))
    {
        daqGetLastError(err, 256);
        wOutput->appendPlainText(QString(tr("Could not query device health due to error: %1")).arg(err));
        return;
    }

    for (i = 0; i < DEVICE_HEALTH_TEMPS; ++i)
    {
        wOutput->appendPlainText(QString(tr("Tmp100-%1 reading is: %2 deg C")).arg(i).arg(temperature[i]));
    }

    QStringList UCDMon_V1;
    UCDMon_V1.push_back("USB5V");
    UCDMon_V1.push_back("D4V3B");
    UCDMon_V1.push_back("D3V8E");
    UCDMon_V1.push_back("D3V8A");
    UCDMon_V1.push_back("D3V8B");
    UCDMon_V1.push_back("D3V8C");
    UCDMon_V1.push_back("D3V8D");
    UCDMon_V1.push_back("D4V3A");
    UCDMon_V1.push_back("USB5V");
    UCDMon_V1.push_back("D2V7");
    UCDMon_V1.push_back("D2V3A");
    UCDMon_V1.push_back("D2V3B");
    UCDMon_V1.push_back("D2V3C");
    UCDMon_V1.push_back("D2V3D");
    UCDMon_V1.push_back("NEG8V");
    UCDMon_V1.push_back("NEG6V");
    UCDMon_V1.push_back("USB5V");
    UCDMon_V1.push_back("A4V3");
    UCDMon_V1.push_back("A3V8");
    UCDMon_V1.push_back("A2V7");
    UCDMon_V1.push_back("A2V3");
    UCDMon_V1.push_back("A7V");
    UCDMon_V1.push_back("A6V");
    UCDMon_V1.push_back("Open");

    QStringList UCDMon_V1_1;
    UCDMon_V1_1.push_back("USB5V");
    UCDMon_V1_1.push_back("D3V8AB");
    UCDMon_V1_1.push_back("D3V8CD");
    UCDMon_V1_1.push_back("D3V8E");
    UCDMon_V1_1.push_back("D4V3");
    UCDMon_V1_1.push_back("NEG8V");
    UCDMon_V1_1.push_back("NEG6V");
    UCDMon_V1_1.push_back("USB5V");
    UCDMon_V1_1.push_back("D2V");
    UCDMon_V1_1.push_back("D1V8A");
    UCDMon_V1_1.push_back("D1V8B");
    UCDMon_V1_1.push_back("D1V8C");
    UCDMon_V1_1.push_back("D1V8D");
    UCDMon_V1_1.push_back("USB5V");
    UCDMon_V1_1.push_back("A4V3");
    UCDMon_V1_1.push_back("A3V8D");
    UCDMon_V1_1.push_back("A3V8B");
    UCDMon_V1_1.push_back("A3V8C");
    UCDMon_V1_1.push_back("A3V8A");
    UCDMon_V1_1.push_back("A7V");
    UCDMon_V1_1.push_back("A6V");
    UCDMon_V1_1.push_back("USB5V");
    UCDMon_V1_1.push_back("A5V_BIAS");
    UCDMon_V1_1.push_back("A2V4");
    UCDMon_V1_1.push_back("A2V1C");
    UCDMon_V1_1.push_back("A2V1B");
    UCDMon_V1_1.push_back("A2V1A");
    UCDMon_V1_1.push_back("A2V1D");

    for (i = 0; i < DEVICE_HEALTH_VOLTS; i++)
    {
        if (daqGetDeviceVersion() == 1.0)
        {
            if (i % 8 == 0 && i < 24)
            {
                wOutput->appendPlainText(QString(tr("The reading of UCD9081-%1 is: ")).arg(i / 8));
            }

            if (i < 24)
            {
                msg = QString(tr("%1: %2 volts")).arg(UCDMon_V1[i]).arg(voltage[i]);
            }
        }
        else if (daqGetDeviceVersion() == 1.1)
        {
            if (i == 0 || i == 7 || i == 13 || i == 21)
            {
                wOutput->appendPlainText(QString(tr("The reading of UCD9081-%1 is: ")).arg(j++));
            }

            msg = QString(tr("%1: %2 volts")).arg(UCDMon_V1_1[i]).arg(voltage[i]);
        }
        else
        {
            msg = tr("ERROR: Unknown device version");
        }

        wOutput->appendPlainText(msg);
    }
}

void AdvDlg::onEnableI2C()
{
    if (daqDoAdvanced(advEnableI2C))
    {
        wOutput->appendPlainText(tr("Enabled I2C"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to enable I2C"));
    }
}

void AdvDlg::onDisableI2C()
{
    if (daqDoAdvanced(advDisableI2C))
    {
        wOutput->appendPlainText(tr("Disabled I2C"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to disable I2C"));
    }
}

void AdvDlg::onResetUSBFifo()
{
    if (daqDoAdvanced(advResetUSBFifo))
    {
        wOutput->appendPlainText(tr("Reset USB FIFO"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to reset USB FIFO"));
    }
}

void AdvDlg::onEmptyFifo()
{
    if (daqDoAdvanced(advEmptyFifo))
    {
        wOutput->appendPlainText(tr("Emptied FIFO"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to empty FIFO"));
    }
}

void AdvDlg::onRxProgBar()
{
    if (daqDoAdvanced(advWriteDevice, eRXPStart, 0, 0))
    {
        wOutput->appendPlainText(tr("Ready for RX start programming"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to set RX for programming"));
    }
}

void AdvDlg::onRxReset()
{
    if (daqDoAdvanced(advResetRxFPGA))
    {
        wOutput->appendPlainText(tr("Reset RX FPGA"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to reset RX FPGA"));
    }
}

void AdvDlg::onResetFPGA()
{
    if (daqDoAdvanced(advResetFPGA))
    {
        wOutput->appendPlainText(tr("Reset Main FPGA"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to reset main FPGA"));
    }
}

void AdvDlg::onResetDCM()
{
    if (daqDoAdvanced(advResetDCM))
    {
        wOutput->appendPlainText(tr("Reset DCM"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to reset DCM"));
    }
}

void AdvDlg::onFlipStart()
{
    if (daqDoAdvanced(advFlipStartBit))
    {
        wOutput->appendPlainText(tr("Flipped start bits"));
    }
    else
    {
        wOutput->appendPlainText(tr("Failed to flip start bits"));
    }
}

void AdvDlg::onChangeAction(int index)
{
    wDeviceAdv->clear();

    if (index < 2)
    {
        wDeviceAdv->addItem("V5 Register", eV5Reg);
        wDeviceAdv->addItem("SEQ SRAM", eSEQSRAM);
        wDeviceAdv->addItem("CT SRAM", eCTSRAM);
        wDeviceAdv->addItem("RX1 Register", eRX1Reg);
        wDeviceAdv->addItem("RX1 SRAM 1", eRX1SRAM1);
        wDeviceAdv->addItem("RX1 SRAM 2", eRX1SRAM2);
        wDeviceAdv->addItem("RX2 Register", eRX2Reg);
        wDeviceAdv->addItem("RX2 SRAM 1", eRX2SRAM1);
        wDeviceAdv->addItem("RX2 SRAM 2", eRX2SRAM2);
        wDeviceAdv->addItem("RX3 Register", eRX3Reg);
        wDeviceAdv->addItem("RX3 SRAM 1", eRX3SRAM1);
        wDeviceAdv->addItem("RX3 SRAM 2", eRX3SRAM2);
        wDeviceAdv->addItem("RX4 Register", eRX4Reg);
        wDeviceAdv->addItem("RX5 SRAM 1", eRX4SRAM1);
        wDeviceAdv->addItem("RX5 SRAM 2", eRX4SRAM2);
        wDeviceAdv->addItem("RX Prog Start", eRXPProg);
        wDeviceAdv->addItem("RX Program", eRXPEnd);
        wDeviceAdv->addItem("RX Register All", eRXRegAll);
        wAddress->setEnabled(true);
    }
    else
    {
        // USB register read/write is selected
        wDeviceAdv->addItem("Control", usbControl);
        wDeviceAdv->addItem("No Timeout", usbNoTimeout);
        wDeviceAdv->addItem("FIFO Status", usbFIFOStatus);
        wDeviceAdv->addItem("Access RAM", usbAccessRAM);
        wDeviceAdv->addItem("RX Reset", usbRXReset);
        wDeviceAdv->addItem("RX Status", usbRXStatus);
        wDeviceAdv->addItem("RX Full", usbRXFull);
        wAddress->setEnabled(false);
    }

    wDeviceAdv->setCurrentIndex(0);
}

void AdvDlg::onSendCommand()
{
    unsigned int device = wDeviceAdv->itemData(wDeviceAdv->currentIndex()).toUInt();
    bool ok;
    unsigned int address = wAddress->text().toInt(&ok, 16), value = wValue->text().toInt(&ok, 16);

    switch (wAction->currentIndex())
    {
    default:
    case 0:
        value = daqDoAdvanced(advReadDevice, device, address);
        if (value != -1)
        {
            wOutput->appendPlainText(tr("Single read success"));
        }
        else
        {
            wOutput->appendPlainText(tr("Single read failed"));
        }
        wValue->setText(QString("%1").arg(value, 8, 16));
        break;
    case 1:
        if (daqDoAdvanced(advWriteDevice, device, address, value))
        {
            wOutput->appendPlainText(tr("Single write success"));
        }
        else
        {
            wOutput->appendPlainText(tr("Single write failed"));
        }
        break;
    case 2:
    {
        auto v = daqDoAdvanced(advReadUsbReg, device);
        if (v != -1)
        {
            wOutput->appendPlainText(tr("Read USB register success"));
        }
        else
        {
            wOutput->appendPlainText(tr("Read USB register failed"));
        }
        wValue->setText(QString("%1").arg(v, 8, 16));
    }
    break;
    case 3:
        if (daqDoAdvanced(advWriteUsbReg, device, (unsigned char)value))
        {
            wOutput->appendPlainText(tr("Write USB register success"));
        }
        else
        {
            wOutput->appendPlainText(tr("Write USB register failed"));
        }
        break;
    }
}
