#include "stdafx.h"
#include "DaqDemo.h"
#include "AdvDlg.h"
#include "TgcDlg.h"
#include "ChannelsDlg.h"

#define MSG_TIMEOUT      3000     // ms
#define DOWNLOAD_REFRESH 50

DaqDemo::DaqDemo(QWidget* parent) : QMainWindow(parent)
{
    int i;

    m_firmwarePath = FIRMWARE_PATH;
    m_examFile = "D:\\daq-b.prm";
    m_downloadFolder = "D:\\daqdata";
    m_datapath = "";

    wProgress = 0;

    m_channels = 128;
    m_channelMap[0] = m_channelMap[1] = m_channelMap[2] = m_channelMap[3] = 0xFFFFFFFF;

    // create a linear tgc curve
    for (i = 2; i >= 0; i--)
    {
        daqTgcSetX(i, (i + 1) * 0.25f);
        daqTgcSetY(i, (i + 1) * 0.25f);
    }

    setupUi(this);
    setupControls();

    m_timer = new QElapsedTimer;
    m_download = new DaqDownload(this);
    m_init = new DaqInit(this);

    connect(this, SIGNAL(sgProgress(int, int)), this, SLOT(onProgress(int, int)));
    connect(m_download, SIGNAL(finished()), this, SLOT(onDownloadFinished()));
    connect(m_init, SIGNAL(finished()), this, SLOT(onInitFinished()));

    daqSetCallback(daqCallback, this);
}

DaqDemo::~DaqDemo()
{
    if (daqIsInitializing())
    {
        daqStopInit();
    }
    if (daqIsDownloading())
    {
        daqStopDownload();
    }

    daqShutdown();
}

// callback function for progress
void DaqDemo::daqCallback(void* prm, int val, ECallbackSources src)
{
    emit((DaqDemo*)prm)->sgProgress(val, src);
}

// create action groups so menus can be checkable
void DaqDemo::setupControls()
{
    QActionGroup* agStart = new QActionGroup(this);
    agStart->addAction(mInternalStart);
    agStart->addAction(mExternalStart);

    QActionGroup* agClock = new QActionGroup(this);
    agClock->addAction(mInternalClk);
    agClock->addAction(mExternalClk);

    QActionGroup* agRunMode = new QActionGroup(this);
    agRunMode->addAction(mFreeRun);
    agRunMode->addAction(mStopFull);

    QActionGroup* agSampling = new QActionGroup(this);
    agSampling->addAction(mSample80);
    agSampling->addAction(mSample40);
    agSampling->addAction(mSample20);
    agSampling->addAction(mSample10);

    wDisplay->init(m_channels, 0);

    wProgress = new QProgressBar();
    wProgress->setMaximumSize(400, QWIDGETSIZE_MAX);
    wTool->addWidget(wProgress);
}

// updates the download status (should be signaled from worker thread into the main thread)
void DaqDemo::onProgress(int val, int source)
{
    if (source == eCbInit)
    {
        wProgress->setValue(val);
    }
    else if (source == eCbDownload)
    {
        if (val % DOWNLOAD_REFRESH == 0)
        {
            QString msg;
            float downloaded = val * 510.0f / (1024 * 1024);
            float rate = (m_timer->elapsed() > 1000) ? downloaded / (m_timer->elapsed() / 1000) : 0;
            msg = QString(tr("%1MB downloaded @ %2MB/s")).arg(downloaded).arg(rate);
            wProgress->setValue(val / DOWNLOAD_REFRESH);
            wStatus->showMessage(msg);
        }
    }
    else if (source == eCbBuffersFull)
    {
        wStatus->showMessage((val == 0) ? tr("Buffers are now full") : tr("Cycling buffers in free run mode"));
    }
}

// allow user to change firmware path
void DaqDemo::onFirmwarePath()
{
    QString dir = QFileDialog::getExistingDirectory(this, tr("Select firmware path"), m_firmwarePath);
    if (!dir.isEmpty())
    {
        m_firmwarePath = dir;
    }
}

// allow user to change exam data file location
void DaqDemo::onExamFile()
{
    QString file = QFileDialog::getOpenFileName(this, tr("Select exam data file"), m_examFile, tr("Data Files (*.prm *.dat)"));
    if (!file.isEmpty())
    {
        m_examFile = file;
    }
}

// allow user to change default download folder
void DaqDemo::onDownloadFolder()
{
    QString dir = QFileDialog::getExistingDirectory(this, tr("Select default download folder"), m_downloadFolder);
    if (!dir.isEmpty())
    {
        m_downloadFolder = dir;
    }
}

// loads the channel mapping window
void DaqDemo::onChannelMap()
{
    ChannelsDlg dlg(m_channelMap);
    if (dlg.exec() == QDialog::Accepted)
    {
        dlg.getChannels(m_channelMap);
    }
}

// let the user set the TGC curve
void DaqDemo::onTgcCurve()
{
    TgcDlg dlg;
    dlg.exec();
}

// starts the download (called by download thread)
void DaqDemo::startInit()
{
    m_initStatus = daqInit(mLoadHighFreq->isChecked()) ? true : false;
}

// initialize the SonixDAQ
void DaqDemo::onInit()
{
    if (daqIsInitializing())
    {
        return;
    }

    wProgress->setRange(0, 100);
    wProgress->setValue(0);

    wStatus->showMessage(tr("Looking for devices..."));
    m_devices.clear();

    char dev[256];
    int index = 0;

    while (daqGetDeviceList(index++, dev, 256))
    {
        m_devices.push_back(dev);
    }

    if (!m_devices.size())
    {
        wStatus->showMessage(tr("Did not find any SonixDAQ devices!"));
        return;
    }
    else
    {
        wStatus->showMessage(QString(tr("Found %1 SonixDAQ devices")).arg(m_devices.size()));
    }

    daqSetFirmwarePath(m_firmwarePath.toAscii().constData());

    if (m_devices.empty())
    {
        return;
    }

    if (daqConnect(0))
    {
        wStatus->showMessage(QString(tr("Connected to device: %1")).arg(m_devices[0].c_str()));
    }
    else
    {
        wStatus->showMessage(QString(tr("Could not connect to device: %1")).arg(m_devices[0].c_str()));
    }

    if (!daqIsConnected())
    {
        return;
    }

    wStatus->showMessage(tr("Attempting to initialize SonixDAQ... this may take a few minutes"));

    m_init->start();
}

// start acquisition
void DaqDemo::onRun()
{
    char err[256];
    bool channelsActive = (m_channelMap[0] != 0) || (m_channelMap[1] != 0) || (m_channelMap[2] != 0) ||
        (m_channelMap[3] != 0) ? true : false;

    if (!daqIsInitialized())
    {
        wStatus->showMessage(tr("SonixDAQ has not been initialized"));
        return;
    }

    if (daqIsRunning())
    {
        wStatus->showMessage(tr("SonixDAQ is already running"));
        return;
    }

    if (!channelsActive)
    {
        wStatus->showMessage(tr("No channels are active, ensure mapping has been completed"));
        return;
    }

    daqRaylinePrms rlprms;
    rlprms.channels = m_channelMap;
    rlprms.numSamples = wParams->item(0, 0)->text().toInt();
    rlprms.lineDuration = wParams->item(1, 0)->text().toInt();
    rlprms.gainDelay = wParams->item(2, 0)->text().toInt();
    rlprms.gainOffset = wParams->item(3, 0)->text().toInt();
    rlprms.rxDelay = wParams->item(4, 0)->text().toInt();
    // decim=2 is 13MHz - not shown in this demo
    rlprms.decimation = (mSample80->isChecked() || mSample40->isChecked()) ? 0 : (mSample20->isChecked() ? 1 : 3);
    rlprms.sampling = mSample80->isChecked() ? 80 : 40;

    daqSequencePrms seqprms;
    seqprms.freeRun = mFreeRun->isChecked() ? 1 : 0;
    seqprms.hpfBypass = (wParams->item(5, 0)->checkState() == Qt::Checked) ? false : true;
    seqprms.divisor = static_cast< unsigned char >(wBuffer->currentIndex());
    seqprms.externalTrigger = mExternalStart->isChecked() ? 1 : 0;
    seqprms.externalClock = mExternalClk->isChecked() ? 1 : 0;
    seqprms.fixedTGC = (wParams->item(7, 0)->checkState() == Qt::Checked) ? true : false;
    seqprms.fixedTGCLevel = wTGC->value();
    seqprms.lnaGain = (wPGAGain->value() - 15) / 3;
    seqprms.pgaGain = (wPGAGain->value() - 21) / 3;
    seqprms.biasCurrent = wBiasCurrent->value();

    if (daqRun(seqprms, rlprms))
    {
        wStatus->showMessage(tr("SonixDAQ sequence is running"));
    }
    else
    {
        daqGetLastError(err, 256);
        wStatus->showMessage(QString(tr("Error running sequence: %1")).arg(err));
    }
}

// stop acquisition
void DaqDemo::onStop()
{
    // see if user tries to stop initialization
    if (daqIsInitializing())
    {
        daqStopInit();
        wStatus->showMessage(tr("Initialization cancelled"));
        return;
    }

    // see if user tries to stop download
    if (daqIsDownloading())
    {
        daqStopDownload();
        wStatus->showMessage(tr("Download cancelled"));
        return;
    }

    if (daqIsRunning())
    {
        wStatus->showMessage(daqStop() ? tr("Stoppped SonixDAQ capture") : tr("Error stopping SonixDAQ capture"));
    }
    else
    {
        wStatus->showMessage(tr("SonixDAQ is not runnning"));
    }
}

// download DAQ data after acquisition
void DaqDemo::onDownload()
{
    int i, szBoard, sz = 0, divisor = wBuffer->currentIndex();
    int full[4];

    // ignore if downloading
    if (daqIsDownloading())
    {
        return;
    }

    if (!daqIsInitialized())
    {
        wStatus->showMessage(tr("SonixDAQ has not been initialized"));
        return;
    }

    // check if running still
    if (daqIsRunning())
    {
        wStatus->showMessage(tr("SonixDAQ is currently running"));
        return;
    }

    // size of board, based on user selection
    szBoard = ((0x80000000 >> divisor) / 255);

    for (i = 0; i < 4; ++i)
    {
        if (m_channelMap[i] && full[i])
        {
            sz += szBoard;
        }
    }

    // get a folder to store the data
    QString dir = QFileDialog::getExistingDirectory(this, tr("Save SonixDAQ Data to Folder"), m_downloadFolder);
    if (dir.isEmpty())
    {
        return;
    }

    m_downloadFolder = dir;

    wProgress->setRange(0, sz / DOWNLOAD_REFRESH);
    wProgress->setValue(0);

    // start the timer for download
    m_timer->restart();
    // start the download thread
    m_download->start();
}

// starts the download (called by download thread)
void DaqDemo::startDownload()
{
    daqDownload(m_downloadFolder.toAscii().constData());
}

// called when the download thread has finished
void DaqDemo::onDownloadFinished()
{
    wProgress->setValue(wProgress->maximum());
    wStatus->showMessage(tr("Download finished"));
}

// called when the init thread has finished
void DaqDemo::onInitFinished()
{
    char err[256];

    if (!m_initStatus)
    {
        daqGetLastError(err, 256);
        wStatus->showMessage(QString(tr("Error initializing SonixDAQ: %1")).arg(err));
    }
    else
    {
        wProgress->setValue(wProgress->maximum());
        wStatus->showMessage(tr("SonixDAQ successfully initialized!"));
        mAdvanced->setEnabled(true);
        aRun->setEnabled(true);
        wDownload->setEnabled(true);
    }
}

// scrolls through the frames of channel data
void DaqDemo::onCine(int index)
{
    loadData(index);
}

// allow user to load data from disk
void DaqDemo::onLoadData()
{
    QString path = QFileDialog::getExistingDirectory(this, tr("Select data path"), m_downloadFolder);
    if (path.isEmpty())
    {
        return;
    }

    m_datapath = path;
    loadData(0, true);
}

// allow data to be loaded from file with a frame offset
void DaqDemo::loadData(int frame, bool init)
{
    int frames = 0;
    std::vector< short > data;

    QDir dir(m_datapath);
    QStringList files = dir.entryList(QStringList("*.daq"));

    for (int i = 0; i < files.count(); i++)
    {
        QString file = dir.path() + "/" + files[i];
        FILE* fp = fopen(file.toLocal8Bit(), "rb");
        if (fp)
        {
            int channel, samples;
            fread(&channel, sizeof(int), 1, fp);
            fread(&frames, sizeof(int), 1, fp);
            fread(&samples, sizeof(int), 1, fp);

            // check for error
            if (frames <= 0 || samples <= 0)
            {
                fclose(fp);
                wStatus->showMessage(tr("Error loading DAQ data"));
                return;
            }

            // if first line, then initialize things
            if (data.empty() && init)
            {
                wDisplay->init(m_channels, samples);
                wCine->setRange(0, frames - 1);
                QString msg = QString(tr("%1 channels: %2 frames @ %3 samples")).arg(files.count()).arg(frames).arg(samples);
                wStatus->showMessage(msg);
            }
            data.resize(samples);

            // load data
            fseek(fp, (sizeof(int) * 3) + (samples * frame), SEEK_SET);
            fread(data.data(), sizeof(short), samples, fp);
            wDisplay->loadChannel(channel, data.data());

            fclose(fp);
        }
    }

    wDisplay->scene()->invalidate();
}

// sets the number of channels to display on screen
void DaqDemo::onChannelDisplay(int value)
{
    wDisplay->setChannelDisplay(value);
}

// sets the amplification of the signals post acquisition
void DaqDemo::onAmp(int value)
{
    wDisplay->setAmp(value);
}

// loads the advanced testing screen
void DaqDemo::onAdvanced()
{
    AdvDlg dlg;
    dlg.exec();
}

DaqDownload::DaqDownload(QObject* parent) : QThread(parent)
{
}

void DaqDownload::run()
{
    setPriority(QThread::TimeCriticalPriority);
    ((DaqDemo*)parent())->startDownload();
}

DaqInit::DaqInit(QObject* parent) : QThread(parent)
{
}

void DaqInit::run()
{
    ((DaqDemo*)parent())->startInit();
}
