#include "stdafx.h"
#include "DaqView.h"

#define bound(y, h) (y < 0 ? ((-y > h) ? -h : y) : ((y > h) ? h : y))

DaqView::DaqView(QWidget* parent) : QGraphicsView(parent)
{
    QGraphicsScene* sc = new QGraphicsScene(this);
    setScene(sc);

    m_channels = 0;
    m_display = m_channels;
    m_samples = 0;
    m_scale = 1.0;
    m_amp = 100;
    m_buffer = 0;
}

DaqView::~DaqView()
{
    if (m_buffer)
    {
        delete[] m_buffer;
    }
}

bool DaqView::init(int channels, int samples)
{
    m_channels = channels;
    m_display = m_channels;
    m_samples = samples;

    if (m_buffer)
    {
        delete[] m_buffer;
        m_buffer = 0;
    }

    if (m_samples)
    {
        m_buffer = new short[m_channels * m_samples];
        memset(m_buffer, 0, m_channels * m_samples * sizeof(short));
    }

    adjustScale();
    return true;
}

void DaqView::loadChannel(int channel, short* data)
{
    if (m_buffer && channel >= 0 && channel < 128)
    {
        memcpy(m_buffer + (channel * m_samples), data, m_samples * sizeof(short));
    }
}

void DaqView::adjustScale()
{
    if (m_samples)
    {
        m_scale = (double)width() / (double)m_samples;
    }
}

void DaqView::setAmp(int amp)
{
    m_amp = amp;
    scene()->invalidate();
}

void DaqView::setChannelDisplay(int chd)
{
    m_display = chd;
    scene()->invalidate();
}

void DaqView::resizeEvent(QResizeEvent* e)
{
    setSceneRect(0, 0, e->size().width(), e->size().height());

    adjustScale();

    QGraphicsView::resizeEvent(e);
}

void DaqView::drawForeground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawForeground(painter, r);

    if (m_buffer)
    {
        int h = height() / m_display;
        double scale = ((double)m_amp / 100.0);
        short* sample = m_buffer;

        painter->setFont(QFont("Arial", h - 4));

        for (int i = 0; i < m_display; i++)
        {
            painter->scale(m_scale, 1);
            painter->setPen(Qt::red);
            QPainterPath pp(QPointF(0, ((i * h) + (h / 2)) + (sample[i * m_samples] * scale)));
            for (int x = 1; x < m_samples; x++)
            {
                auto y = bound((sample[i * m_samples + x] * scale), (h / 2));
                pp.lineTo(x, ((i * h) + (h / 2)) + y);
            }
            painter->drawPath(pp);

            painter->setPen(Qt::black);
            painter->drawRect(0, i * h, m_samples - 1, h - 1);

            painter->resetTransform();
            painter->drawText(1, ((i + 1) * h) - 2, QString("%1").arg(i + 1));
        }
    }
}
