#include "stdafx.h"
#include "FluctusDemo.h"

#define MSG_TIMEOUT         3000 // ms
#define CORSCIENCE_BASELINE 154

static QActionGroup* agPort = 0;

FluctusDemo::FluctusDemo(QWidget* parent) : QMainWindow(parent)
{
    setupUi(this);
    setupControls();

    fluctusSetCallback(onData, this);

    QObject::connect(this, SIGNAL(showMessage(QString)), this, SLOT(onShowMessage(QString)));
}

FluctusDemo::~FluctusDemo()
{
    fluctusStop();
    fluctusShutdown();
}

// create action groups so menus can be checkable
void FluctusDemo::setupControls()
{
    QActionGroup* agVersion = new QActionGroup(this);
    agVersion->addAction(mCorscience);
    agVersion->addAction(mLT);

    agPort = new QActionGroup(this);
    agPort->addAction(mPort1);
    agPort->addAction(mPort2);
    agPort->addAction(mPort3);
    agPort->addAction(mPort4);
    agPort->addAction(mPort5);
    agPort->addAction(mPort6);

    mPort1->setData(1);
    mPort2->setData(2);
    mPort3->setData(3);
    mPort4->setData(4);
    mPort5->setData(5);
    mPort6->setData(6);

    wDisplay->setAmp(wAmp->value());
    wBaseline->setRange(1, wDisplay->height());
    wBaseline->setValue(wDisplay->height() / 2);
}

// initialize the ultrasound engine, populate probes, load lists
void FluctusDemo::init()
{
    int port = agPort->checkedAction()->data().toInt();

    if (fluctusIsRunning())
    {
        fluctusStop();
    }

    // try and initialize the ECG
    if (!fluctusInit(mCorscience->isChecked() ? ecgVersionCorscience : ecgVersionLT, port))
    {
        wStatus->showMessage(tr("Could not intitialize ecg"), MSG_TIMEOUT);
        return;
    }

    wStatus->showMessage(tr("Successfully intitialized ecg"), MSG_TIMEOUT);
    aRun->setEnabled(true);
    aStop->setEnabled(true);

    updateParams();
}

void FluctusDemo::resizeEvent(QResizeEvent* e)
{
    QMainWindow::resizeEvent(e);

    wBaseline->setRange(1, wDisplay->height());
    wBaseline->setValue(wDisplay->height() / 2);

    wScroll->setRange(0, wDisplay->getScrollSize());
    wScroll->setValue(wScroll->maximum());
    wDisplay->setScroll(wScroll->maximum(), fluctusIsRunning() ? false : true);
}

// callback for new data
int FluctusDemo::onData(void* prm, int info, int data)
{
    QString leads, pace, arr;
    FluctusDemo* wnd = (FluctusDemo*)prm;

    if (info == ecgNoCableDetected)
    {
        leads = tr("No Cable Detected");
    }
    else if (info == ecgLeadOff)
    {
        leads = tr("Lead Off");
    }
    else if (info == ecgLeadReconnected)
    {
        leads = tr("Leads Connected");
    }
    else if (info == ecgPulseRate)
    {
        wnd->wDisplay->setPulseRate(data);
    }
    else if (info == ecgPulseRateTooHigh)
    {
        wnd->wDisplay->setPulseRate(BPM_HIGH);
    }
    else if (info == ecgArrhythmia)
    {
        arr = QString(tr("Arrhythmia: %1")).arg(data);
    }
    else if (info == ecgPacemakerDetected)
    {
        pace = tr("Pacemaker Detected");
    }
    else if (info == ecgWFDetect)
    {
        wnd->wDisplay->markPeak();
    }
    else if (info == ecgLeadData)
    {
        wnd->wDisplay->add(data - CORSCIENCE_BASELINE);
    }

    // note that we use a 'signal' to get out of the worker thread, otherwise if we make the status bar
    // update outside of the main thread, the program may crash
    if (!pace.isEmpty())
    {
        emit wnd->showMessage(pace);
    }
    else if (!arr.isEmpty())
    {
        emit wnd->showMessage(arr);
    }
    else if (!leads.isEmpty())
    {
        emit wnd->showMessage(leads);
    }

    return true;
}

// slot for updating status bar when worker threads need to do so
void FluctusDemo::onShowMessage(QString msg)
{
    wStatus->showMessage(msg, MSG_TIMEOUT);
}

// start sampling
void FluctusDemo::run()
{
    wDisplay->clear();
    wDisplay->setScroll(0, false);
    wScroll->setEnabled(false);

    if (!fluctusStart())
    {
        wStatus->showMessage(tr("Could not start ECG"), MSG_TIMEOUT);
    }
    else
    {
        wStatus->showMessage(tr("ECG acquiring data"), MSG_TIMEOUT);
    }
}

// stop sampling
void FluctusDemo::stop()
{
    wScroll->setRange(0, wDisplay->getScrollSize());
    wScroll->setValue(wScroll->maximum());
    wScroll->setEnabled(true);
    wDisplay->setScroll(wScroll->maximum(), true);

    if (!fluctusStop())
    {
        wStatus->showMessage(tr("Could not stop ECG"), MSG_TIMEOUT);
    }
    else
    {
        wStatus->showMessage(tr("ECG stopped"), MSG_TIMEOUT);
    }
}

void FluctusDemo::onAmp(int value)
{
    wDisplay->setAmp(value);
}

void FluctusDemo::onBaseline(int value)
{
    wDisplay->setBaseline(value);
}

void FluctusDemo::onScroll(int value)
{
    wDisplay->setScroll(value, true);
}

void FluctusDemo::onParamCheck(int row, int)
{
    if (row == 6)
    {
        wDisplay->setInvert(wParams->item(6, 0)->checkState() == Qt::Checked);
    }
    else if (row == 7)
    {
        wDisplay->setShowPeaks(wParams->item(7, 0)->checkState() == Qt::Checked);
    }
}

void FluctusDemo::inc()
{
    if (!wParams->currentItem())
    {
        return;
    }

    if (wParams->currentItem()->row() == 1)
    {
        changeSamplingRate(true);
    }
    else if (wParams->currentItem()->row() == 2)
    {
        changeGain(true);
    }
    else if (wParams->currentItem()->row() == 3)
    {
        changeLead(true);
    }
    else if (wParams->currentItem()->row() == 4)
    {
        changeFilter(true);
    }
}

void FluctusDemo::dec()
{
    if (!wParams->currentItem())
    {
        return;
    }

    if (wParams->currentItem()->row() == 1)
    {
        changeSamplingRate(false);
    }
    else if (wParams->currentItem()->row() == 2)
    {
        changeGain(false);
    }
    else if (wParams->currentItem()->row() == 3)
    {
        changeLead(false);
    }
    else if (wParams->currentItem()->row() == 4)
    {
        changeFilter(false);
    }
}

void FluctusDemo::changeSamplingRate(bool inc)
{
    ecgSamplingRate rate;

    rate = fluctusGetSamplingRateInfo();
    rate =
        inc ? (rate == ecgSamplingRateMax) ? ecgSamplingRateLow : (ecgSamplingRate)(rate + 1) : (rate ==
                                                                                                 ecgSamplingRateLow) ? ecgSamplingRateMax : (
            ecgSamplingRate)(rate - 1);
    fluctusSetSamplingRate(rate);

    updateParams();
}

void FluctusDemo::changeGain(bool inc)
{
    ecgGainLevel gain;

    gain = fluctusGetGainInfo();
    gain =
        inc ? (gain == ecgGainHigh) ? ecgGainLow : (ecgGainLevel)(gain + 1) : (gain == ecgGainLow) ? ecgGainHigh : (ecgGainLevel)(gain - 1);
    fluctusSetGain(gain);

    updateParams();
}

void FluctusDemo::changeLead(bool inc)
{
    ecgLead lead;

    lead = fluctusGetLeadInfo();
    lead = inc ? ((lead == ecgLeadSTD) ? ecgLeadI : (ecgLead)(lead + 1)) : ((lead == ecgLeadI) ? ecgLeadSTD : (ecgLead)(lead - 1));
    fluctusSetLead(lead);

    updateParams();
}

void FluctusDemo::changeFilter(bool inc)
{
    ecgFilterType filt;

    filt = fluctusGetFilterInfo();
    filt =
        inc ? (filt == ecgNotchFilter50Hz) ? ecgFilterNone : (ecgFilterType)(filt + 1) : (filt ==
                                                                                          ecgFilterNone) ? ecgNotchFilter50Hz : (
            ecgFilterType)(filt - 1);
    fluctusSetFilter(filt);

    updateParams();
}

void FluctusDemo::updateParams()
{
    ecgSamplingRate rate;
    ecgGainLevel gain;
    ecgLead lead;
    ecgFilterType filt;

    rate = fluctusGetSamplingRateInfo();
    gain = fluctusGetGainInfo();
    lead = fluctusGetLeadInfo();
    filt = fluctusGetFilterInfo();

    wParams->item(1, 0)->setText(QString("%1 Hz").arg(fluctusGetSamplingRate()));

    if (gain == ecgGainLow)
    {
        wParams->item(2, 0)->setText("Low");
    }
    else if (gain == ecgGainMed)
    {
        wParams->item(2, 0)->setText("Med");
    }
    else if (gain == ecgGainHigh)
    {
        wParams->item(2, 0)->setText("High");
    }

    if (lead == ecgLeadI)
    {
        wParams->item(3, 0)->setText("I");
    }
    else if (lead == ecgLeadII)
    {
        wParams->item(3, 0)->setText("II");
    }
    else if (lead == ecgLeadIII)
    {
        wParams->item(3, 0)->setText("III");
    }
    else if (lead == ecgLeadSTD)
    {
        wParams->item(3, 0)->setText("GND");
    }

    if (filt == ecgFilterNone)
    {
        wParams->item(4, 0)->setText("None");
    }
    else if (filt == ecgHamming25Hz)
    {
        wParams->item(4, 0)->setText("Hamming");
    }
    else if (filt == ecgBlackman25Hz)
    {
        wParams->item(4, 0)->setText("Blackman");
    }
    else if (filt == ecgNotchFilter50Hz)
    {
        wParams->item(4, 0)->setText("Notch");
    }
}
