////////////////////////////////////////////////////////////////////////////////
///@file fluctus_def.h
////////////////////////////////////////////////////////////////////////////////
#pragma once

////////////////////////////////////////////////////////////////////////////////
/// ECG Version Definition
////////////////////////////////////////////////////////////////////////////////
enum ecgVersion
{
    ////////////////////////////////////////////////////////////////////////////////
    /// MCC ECG
    /// \par
    /// Supports:
    /// \li Selectable lead retrieval (II and III)
    /// \li 3 gain levels
    /// \li 4 sampling rates
    /// \li 2 notch filters
    /// \li Lead off / reconnect detection
    /// \li Pacemaker detection
    /// \li Pulse rate values
    ////////////////////////////////////////////////////////////////////////////////
    ecgVersionMCC = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// L&T ECG
    /// \par
    /// Supports:
    /// \li Selectable lead retrieval (I, II, III, STD)
    /// \li 3 Gain levels
    /// \li Fixed 200 Hz sampling rate
    /// \li Lead off / cable off detection
    /// \li Pulse rate values
    /// \li Arrhythmia values
    ////////////////////////////////////////////////////////////////////////////////
    ecgVersionLT = 2,
    ////////////////////////////////////////////////////////////////////////////////
    /// Corscience ECG
    /// \par
    /// Supports:
    /// \li Selectable lead retrieval (II and III)
    /// \li Programmable sampling rates
    /// \li Lead off / cable off detection
    /// \li Pulse rate values
    /// \li Arrhythmia values
    ////////////////////////////////////////////////////////////////////////////////
    ecgVersionCorscience = 3
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Information Codes
////////////////////////////////////////////////////////////////////////////////
enum ecgInformation
{
    ////////////////////////////////////////////////////////////////////////////////
    /// Used internally.
    ////////////////////////////////////////////////////////////////////////////////
    ecgUnknown = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead data message
    /// \par
    /// \b MCC Version
    /// \li Data: lead II or lead III data
    /// \par
    /// \b L&T Version
    /// \li Data: selected lead data
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadData = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// Pulse rate message
    /// \par
    /// \b MCC & L&T Versions
    /// \li Data: pulse rate in Hz
    ////////////////////////////////////////////////////////////////////////////////
    ecgPulseRate,
    ////////////////////////////////////////////////////////////////////////////////
    /// Pulse rate is too high message
    /// \par
    /// \b MCC Version
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgPulseRateTooHigh,
    ////////////////////////////////////////////////////////////////////////////////
    /// Arrhythmia message
    /// \par
    /// \b L&T Version
    /// \li Data: arrhythmia value
    ////////////////////////////////////////////////////////////////////////////////
    ecgArrhythmia,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead off message
    /// \par
    /// \b MCC & L&T Versions
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadOff,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead reconnected message
    /// \par
    /// \b MCC Version
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadReconnected,
    ////////////////////////////////////////////////////////////////////////////////
    /// No cable connected message
    /// \par
    /// \b L&T Version
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgNoCableDetected,
    ////////////////////////////////////////////////////////////////////////////////
    /// Pacemaker detected message
    /// \par
    /// \b MCC Version
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgPacemakerDetected,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead and gain information message
    /// \par
    /// \b L&T Version
    /// \li Data: selected lead (high 16 bits), and selected gain (low 16 bits)
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadGainInfo,
    ////////////////////////////////////////////////////////////////////////////////
    /// Version information message
    /// \par
    /// \b L&T Version
    /// \li Data: ecg version
    ////////////////////////////////////////////////////////////////////////////////
    ecgVersionInfo,
    ////////////////////////////////////////////////////////////////////////////////
    /// Information code passed back when the previous lead data sent back was
    /// detected as the requested part of the waveform to be detected (QRS+PT)
    /// \par
    /// \b MCC & L&T Versions
    /// \li Data: unused
    ////////////////////////////////////////////////////////////////////////////////
    ecgWFDetect
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Lead Selection
/// \li Supports: MCC (leads II and III), L&T (all leads), and Corscience (leads II and III)
////////////////////////////////////////////////////////////////////////////////
enum ecgLead
{
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead I
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadI = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead II
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadII = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead III
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadIII,
    ////////////////////////////////////////////////////////////////////////////////
    /// Lead STD
    ////////////////////////////////////////////////////////////////////////////////
    ecgLeadSTD
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Gain Selection
/// \li Supports: MCC & L&T Versions
////////////////////////////////////////////////////////////////////////////////
enum ecgGainLevel
{
    ////////////////////////////////////////////////////////////////////////////////
    /// Low Gain
    /// \li MCC: 5 uV/Bit
    /// \li L&T: 0.3 mV
    ////////////////////////////////////////////////////////////////////////////////
    ecgGainLow = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// Medium Gain
    /// \li MCC: 10 uV/Bit
    /// \li L&T: 1 mV
    ////////////////////////////////////////////////////////////////////////////////
    ecgGainMed = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// High Gain
    /// \li MCC: 20 uV/Bit
    /// \li L&T: 2 mV
    ////////////////////////////////////////////////////////////////////////////////
    ecgGainHigh
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Sampling Rate Selection
/// \li Supports: MCC & Corscience Versions
////////////////////////////////////////////////////////////////////////////////
enum ecgSamplingRate
{
    ////////////////////////////////////////////////////////////////////////////////
    /// Low Sampling Rate
    /// \li MCC: 37.5 Hz
    /// \li Corscience: 100 Hz
    ////////////////////////////////////////////////////////////////////////////////
    ecgSamplingRateLow = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// Medium Sampling Rate
    /// \li MCC: 75 Hz
    /// \li Corscience: 200 Hz
    ////////////////////////////////////////////////////////////////////////////////
    ecgSamplingRateMed = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// High Sampling Rate
    /// \li MCC: 150 Hz
    /// \li Corscience: 500 Hz
    ////////////////////////////////////////////////////////////////////////////////
    ecgSamplingRateHigh,
    ////////////////////////////////////////////////////////////////////////////////
    /// Max Sampling Rate
    /// \li MCC: 300 Hz
    /// \li Corscience: 1000 Hz
    ////////////////////////////////////////////////////////////////////////////////
    ecgSamplingRateMax
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Notch Filter Selection
/// \li Supports: Corscience Version
////////////////////////////////////////////////////////////////////////////////
enum ecgFilterType
{
    ecgFilterNone = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// 25 Hz Hamming filter
    ////////////////////////////////////////////////////////////////////////////////
    ecgHamming25Hz = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// 25 Hz Blackman filter
    ////////////////////////////////////////////////////////////////////////////////
    ecgBlackman25Hz,
    ////////////////////////////////////////////////////////////////////////////////
    /// 50 Hz notch filter
    ////////////////////////////////////////////////////////////////////////////////
    ecgNotchFilter50Hz
};

////////////////////////////////////////////////////////////////////////////////
/// ECG Peak Detection Selection
/// \li Supports: MCC & L&T Versions
////////////////////////////////////////////////////////////////////////////////
enum ecgPeakDetection
{
    ////////////////////////////////////////////////////////////////////////////////
    /// No detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectNone = 0,
    ////////////////////////////////////////////////////////////////////////////////
    /// P Wave detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectPWave = 1,
    ////////////////////////////////////////////////////////////////////////////////
    /// Q Wave detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectQWave = 2,
    ////////////////////////////////////////////////////////////////////////////////
    /// R Wave detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectRWave = 3,
    ////////////////////////////////////////////////////////////////////////////////
    /// S Wave detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectSWave = 4,
    ////////////////////////////////////////////////////////////////////////////////
    /// T Wave detection
    ////////////////////////////////////////////////////////////////////////////////
    ecgDetectTWave = 3
};

////////////////////////////////////////////////////////////////////////////////
/// Callback used when ECG data or a message is ready.
///
///@param           prm Custom user parameter that is passed from the callback.
///@param           info The information code.
///@param           data_a The data associated with the information code.
///@return          A boolean value. Reserved for future use.
///@sa              Ecg::setCallback()
///
////////////////////////////////////////////////////////////////////////////////
typedef int (*ECG_CALLBACK)(void* prm, int info, int data);
