#include "StdAfx.h"

#define B_IMAGEWIDTH  640
#define B_IMAGEHEIGHT 480
#define M_IMAGEWIDTH  800
#define M_IMAGEHEIGHT 300
#define CINE_SIZE     (64 * 1024 * 1024)
#define swap(a, b) { temp = (a); (a) = (b); (b) = temp; }

void testElectronics();
bool selectProbe();
bool selectMode();
bool toggleMotor();
bool changeParam();
bool saveImage();
bool saveData(unsigned int blockindex);
void porta_error(const char* err);
void invertImage(unsigned int* pData, int w, int h);
bool writeBitmap(const char* path, unsigned char* data, int w, int h, int bitdepth);

bool probeSelected = false;
bool modeLoaded = false;
int m_modeloaded = 0;

void usage(char* prog)
{
    printf("\n");
    printf("usage: %s [options]\n", prog);
    printf("options:\n");
    printf("-fw PATH set the firmware path as specified by PATH\n");
    printf("-pci2    use sonix op/sp/rp pci card [default]\n");
    printf("-pci3    use sonix mdp/rp+ pci card (4 BNC connectors)\n");
    printf("-pci4    use sonix mdp/rp+ pci card (4 BNC connectors)\n");
    printf("-usm2    use sonix op/sp/rp ultrasound module [default]\n");
    printf("-usm3    use mdp/rp+ ultrasound module\n");
    printf("-usm4    use mdp/rp+ ultrasound module\n");
    printf("-hv      use high-voltage board (usm2 only)\n");
    printf("-test    run tests\n");
    printf("-help    display this help\n");
    printf("notes:\n");
    printf("-default firmware path is '%s'\n", FIRMWARE_PATH);
}

// program entry point
int main(int argc, char* argv[])
{
    char sel;
    int i;
    bool test = false;
    char fwpath[1024];

    strcpy(fwpath, FIRMWARE_PATH);

    int pci = 3, usm = 4, hv = 0, ddr = 0, channels = 64;

    for (i = 1; i < argc; i++)
    {
        if (strcmp(argv[i], "-pci2") == 0)
        {
            pci = 2;
        }
        else if (strcmp(argv[i], "-pci3") == 0)
        {
            pci = 3;
        }
        else if (strcmp(argv[i], "-pci4") == 0)
        {
            pci = 4;
        }
        else if (strcmp(argv[i], "-usm2") == 0)
        {
            usm = 2;
        }
        else if (strcmp(argv[i], "-usm3") == 0)
        {
            usm = 3;
        }
        else if (strcmp(argv[i], "-usm4") == 0)
        {
            usm = 4;
        }
        else if (strcmp(argv[i], "-hv") == 0)
        {
            hv = 1;
        }
        else if (strcmp(argv[i], "-test") == 0)
        {
            test = true;
        }
        else if (strcmp(argv[i], "-fw") == 0)
        {
            strcpy(fwpath, argv[i + 1]);
            i++;
        }
        else
        {
            usage(argv[0]);
            return 0;
        }
    }

    strcat(fwpath, "\\");

    if (!portaInit(CINE_SIZE, fwpath, SETTINGS_PATH, LICENSE_PATH, LUT_PATH, usm, pci, hv, ddr, channels))
    {
        porta_error("system could not be intialized.");
        return -1;
    }

    // test electronics if specified
    if (test)
    {
        testElectronics();
    }

    // display a selection for the user
    for (;;)
    {
        fprintf(stdout, "\nmake a selection:\n\n");
        fprintf(stdout, "1) select a transducer\n");
        fprintf(stdout, "2) select an imaging mode\n");
        fprintf(stdout, "3) run\n");
        fprintf(stdout, "4) stop\n");
        fprintf(stdout, "5) save image\n");
        fprintf(stdout, "6) save cine buffer\n");
        fprintf(stdout, "7) change parameter\n");
        fprintf(stdout, "8) toggle motor\n");
        fprintf(stdout, "X) shutdown\n");

        // read the user selection
        scanf("%c", &sel);

        fflush(stdin);

        switch (sel)
        {
        case '1': selectProbe();
            break;

        case '2': selectMode();
            break;

        case '3':
        {
            if (!modeLoaded)
            {
                porta_error("no imaging mode selected.");
            }
            else
            {
                portaRunImage();
            }
        }
        break;
        case '4':
        {
            if (portaIsImaging())
            {
                portaStopImage();
                int framecnt = -1;
                if (portaGetCurrentMode() == PwMode)
                {
                    framecnt = portaGetDisplayFrameCount(1);
                }
                else
                {
                    framecnt = portaGetDisplayFrameCount(0);
                }
                fprintf(stdout, "frames available in cine: %d\n", framecnt);
                portaGetFrameCount(0);
            }
            else
            {
                porta_error("imaging is not running.");
            }
        }
        break;
        case '5':
        {
            saveImage();
        }
        break;
        case '6':
        {
            saveData(0);
            if (portaGetCurrentMode() == BiplaneMode)
            {
                saveData(1);
            }
        }
        break;
        case '7': changeParam();
            break;
        case '8': toggleMotor();
            break;
        case 'X':
        case 'x':
            fprintf(stdout, "\nexiting...\n");
            goto goodbye;
        }
    }

goodbye:

    portaStopImage();
    portaShutdown();

    return 0;
}

void testElectronics()
{
    double pv[16], nv[16], pav[16], nav[16], tol;
    int i, tmout;

    printf("\n");
    printf("probe connectors readback:\n");

    for (i = 0; i < 3; i++)
    {
        printf("connector %d: %d\n", i, portaGetProbeID(i));
    }

    printf("\n");
    printf("reading power values, please wait\n");

    portaReadPowerValues(pv, nv, pav, nav, tmout, tol);
    printf("tolerance: %.2f%%, tmout: %dms\n", tol, tmout);
    for (i = 0; i < 16; i++)
    {
        printf("p: %.2f [%.2f], n: %.2f [%.2f]\n", pv[i], pav[i], nv[i], nav[i]);
    }
    printf("\n");
    printf("performing component tests:\n");
    for (i = 0; i <= 15; i++)
    {
        if (!portaTestElectronicComponent(i))
        {
            printf("failed test #%d\n", i);
        }
        else
        {
            printf("passed test #%d\n", i);
        }
    }
    printf("\n");
    printf("testing finished:\n");
    printf("\n");
}

// writes an error message to screen
void porta_error(const char* err)
{
    fprintf(stderr, "Error: ");
    fprintf(stderr, err);
    fprintf(stderr, "\n");
}

// runs the user through the process of selecting a transducer
bool selectProbe()
{
    int i, sel, codes[3], currentcode;
    char str[1024];

    // get the transducer id's plugged into each port
    for (i = 0; i < 3; i++)
    {
        codes[i] = portaGetProbeID(i);
    }

    // display the transducer information
    fprintf(stdout, "select a transducer:\n");
    for (i = 0; i < 3; i++)
    {
        currentcode = codes[i];
        portaGetProbeName(str, 1024, currentcode);

        if (currentcode != 31)
        {
            fprintf(stdout, "%d) %s\n", i + 1, str);
        }
        else
        {
            fprintf(stdout, "%d) no transducer connected\n", i + 1);
        }
    }

    scanf("%d", &sel);

    if (sel < 1 || sel > 3 || codes[sel - 1] == 31)
    {
        porta_error("invalid selection.");
        return false;
    }

    // activate the selected transducer
    portaGetProbeName(str, 1024, codes[sel - 1]);
    fprintf(stdout, "\nselecting: %s\n", str);
    if (portaSelectProbe(codes[sel - 1]))
    {
        portaActivateProbeConnector(sel - 1);
    }

    // find and load master preset
    if (!portaFindMasterPreset(str, 1024, codes[sel - 1]))
    {
        porta_error("master preset could not be found.");
        return false;
    }

    //load the master preset
    if (!portaLoadPreset(str))
    {
        porta_error("master preset could not be loaded.");
        return false;
    }

    probeSelected = true;

    return true;
}

// runs the user through the process of selecting an imaging mode
bool selectMode()
{
    if (!probeSelected)
    {
        porta_error("no probe selected.");
        return false;
    }

    int sel, mode = BMode, probeID = portaGetCurrentProbeID();

    fprintf(stdout, "\nmake a selection:\n\n");
    fprintf(stdout, "1) B mode\n");
    fprintf(stdout, "2) M mode\n");
    fprintf(stdout, "3) PW Doppler\n");
    fprintf(stdout, "4) Color Doppler\n");
    //biplane enabled probe
    if (probeID == 14)
    {
        fprintf(stdout, "5) Biplane Mode\n");
    }

    scanf("%d", &sel);

    if (sel < 1 || sel > 5)
    {
        porta_error("invalid selection.");
        return false;
    }

    switch (sel)
    {
    case 1: mode = BMode;
        break;
    case 2: mode = MMode;
        break;
    case 3: mode = PwMode;
        break;
    case 4: mode = ColourMode;
        break;
    case 5: mode = BiplaneMode;
        break;
    }
    m_modeloaded = sel;

    // create mode
    if (!portaInitMode((imagingMode)mode))
    {
        porta_error("mode could not be initialized. check licenses or probe setup");
        return false;
    }

    portaSetDisplayDimensions(0, B_IMAGEWIDTH, B_IMAGEHEIGHT);

    if (mode == MMode || mode == PwMode)
    {
        portaSetDisplayDimensions(1, M_IMAGEWIDTH, M_IMAGEHEIGHT);
    }
    else if (mode == BiplaneMode)
    {
        portaSetDisplayDimensions(1, B_IMAGEWIDTH, B_IMAGEHEIGHT);
    }

    modeLoaded = true;

    return true;
}

bool toggleMotor()
{
    probeInfo nfo;
    portaGetProbeInfo(nfo);

    if (!nfo.motorized)
    {
        porta_error("probe does not support motor control.");
        return false;
    }

    portaSetParamI(prmMotorStatus, portaGetParam(prmMotorStatus) ? 0 : 1);

    return true;
}

// runs the user through the process of changing a parameter
bool changeParam()
{
    if (!probeSelected)
    {
        porta_error("no valid preset loaded.");
        return false;
    }

    int sel;

    fprintf(stdout, "\nmake a selection:\n\n");
    fprintf(stdout, "1) inc depth\n");
    fprintf(stdout, "2) dec depth\n");
    fprintf(stdout, "3) inc gain\n");
    fprintf(stdout, "4) dec gain\n");
    fprintf(stdout, "5) list values\n");

    scanf("%d", &sel);

    if (sel < 1 || sel > 5)
    {
        porta_error("invalid selection.");
        return false;
    }

    switch (sel)
    {
    case 1: portaCycleParam(prmBImageDepth, 1);
        break;
    case 2: portaCycleParam(prmBImageDepth, 0);
        break;
    case 3: portaCycleParam(prmBGain, 1);
        break;
    case 4: portaCycleParam(prmBGain, 0);
        break;
    }

    int depth, gain;
    portaGetParamI(prmBImageDepth, depth);
    fprintf(stdout, "\nimage depth = %.1f cm", (double)depth / 10.0);
    portaGetParamI(prmBGain, gain);
    fprintf(stdout, "\ngain = %d", gain);
    fprintf(stdout, "\n");

    return true;
}

// dumps the cine buffer to a file
bool saveData(unsigned int blockindex)
{
    char path[1024];
    int numFrames, frameSize;
    FILE* fp;

    numFrames = portaGetFrameCount(blockindex);
    frameSize = portaGetFrameSize(0);

    if (numFrames < 1)
    {
        fprintf(stderr, "no frames have been acquired\n");
        return false;
    }

    printf("enter a filename: ");
    scanf("%s", path);

    fp = fopen(path, "wb+");
    if (!fp)
    {
        fprintf(stderr, "could not store data to specified path\n");
        return false;
    }

    fwrite(portaGetCineStart(blockindex), frameSize, numFrames, fp);

    fclose(fp);

    fprintf(stdout, "successfully stored data\n");

    return true;
}

// stores a bitmap image to hard-disk
bool saveImage()
{
    char path[1024];

    if (!modeLoaded)
    {
        porta_error("no imaging mode has been selected.");
        return false;
    }

    int running = portaIsImaging();

    int mode = portaGetCurrentMode();

    if (running)
    {
        portaStopImage();
    }

    if (portaGetDisplayFrameCount(0) <= 1)
    {
        porta_error("no frames have been acquired.");
        return false;
    }

    unsigned char* imgBuffer = new unsigned char[B_IMAGEWIDTH * B_IMAGEHEIGHT * sizeof(unsigned int)];

    if (mode == ColourMode)
    {
        portaGetColorImage(0, imgBuffer);
    }
    else if (mode == BiplaneMode)
    {
        portaGetBwImage(0, imgBuffer, true);
        fprintf(stdout, "enter path to save file 1: ");
        scanf("%s", path);
        writeBitmap(path, imgBuffer, B_IMAGEWIDTH, B_IMAGEHEIGHT, 32);

        portaGetBwImage(1, imgBuffer, true);
        fprintf(stdout, "enter path to save file 2: ");
        scanf("%s", path);
        writeBitmap(path, imgBuffer, B_IMAGEWIDTH, B_IMAGEHEIGHT, 32);
    }
    else
    {
        // to save postscan b data
        portaGetBwImage(0, imgBuffer, true);
        fprintf(stdout, "enter path to save postscan b file: ");
        scanf("%s", path);
        writeBitmap(path, imgBuffer, B_IMAGEWIDTH, B_IMAGEHEIGHT, 32);

        // get save prescan b data
        /*
           int w,h;
           portaGetPrescanBDimensions(0, w, h);
           unsigned char* prescanimgBuffer = new unsigned char[w * h * sizeof(unsigned int)];
           portaGetBwImagePrescan(0, prescanimgBuffer);
           fprintf(stdout, "enter path to save prescan b file: ");
           scanf("%s", path);
           FILE *fp = fopen(path, "wb+");
           if (!fp)
           {
            fprintf(stderr, "could not store data to specified path\n");
            return false;
           }
           fwrite(prescanimgBuffer, w*h, 1, fp);
           fclose(fp);
         */
    }

    if (running)
    {
        portaRunImage();
    }

    delete[] imgBuffer;

    return true;
}

// inverts image data, so that the bitmap can be displayed properly
void invertImage(unsigned int* pData, int w, int h)
{
    int x, y, temp;
    unsigned int* p = (unsigned int*)pData;

    // flip the image on the X-axis
    for (y = 0; y < h / 2; y++)
    {
        for (x = 0; x < w; x++)
        {
            swap(p[y * w + x], p[((h - 1 - y) * w) + x]);
        }
    }
}

// write a bitmap file to a specified location
bool writeBitmap(const char* path, unsigned char* data, int w, int h, int bitdepth)
{
    FILE* fp = fopen(path, "wb");
    if (!fp)
    {
        porta_error("file could not be created.");
        return false;
    }

    // since the data gets flipped when written to bitmap, pre-flip the image
    invertImage((unsigned int*)data, w, h);

    BITMAPINFOHEADER BMIH;
    BMIH.biSize = sizeof(BITMAPINFOHEADER);
    BMIH.biBitCount = (unsigned short)bitdepth;
    BMIH.biPlanes = 1;
    BMIH.biCompression = BI_RGB;
    BMIH.biWidth = w;
    BMIH.biHeight = h;
    BMIH.biSizeImage = ((((BMIH.biWidth * BMIH.biBitCount) + 31) & ~31) >> 3) * BMIH.biHeight;

    BITMAPFILEHEADER bmfh;
    int nBitsOffset = sizeof(BITMAPFILEHEADER) + BMIH.biSize;
    LONG lImageSize = BMIH.biSizeImage;
    LONG lFileSize = nBitsOffset + lImageSize;
    bmfh.bfType = 'B' + ('M' << 8);
    bmfh.bfOffBits = nBitsOffset;
    bmfh.bfSize = lFileSize;
    bmfh.bfReserved1 = bmfh.bfReserved2 = 0;
    //Write the bitmap file header
    fwrite(&bmfh, 1, sizeof(BITMAPFILEHEADER), fp);
    //And then the bitmap info header
    fwrite(&BMIH, 1, sizeof(BITMAPINFOHEADER), fp);
    //Finally, write the image data itself
    //-- the data represents our drawing
    fwrite(data, 1, lImageSize, fp);

    fclose(fp);

    return true;
}
