#include "stdafx.h"
#include "Propello.h"
#include "SaveDlg.h"

#ifndef FIRMWARE_PATH
    #define FIRMWARE_PATH "../../fw/"
#endif

#ifndef SETTINGS_PATH
    #define SETTINGS_PATH "../../dat/"
#endif

#define PRESETS_PATH (SETTINGS_PATH "presets/imaging/")

#define MSG_TIMEOUT  3000    // ms
#define CINE_SIZE    512     // MB

Propello::Propello(QWidget* parent) : QMainWindow(parent)
{
    setupUi(this);
    setupControls();

    m_firmwarePath = FIRMWARE_PATH;
    m_settingsPath = SETTINGS_PATH;
    m_cineSize = CINE_SIZE;
    m_motorstart = 0;
    m_motorpos = 0.0;

    connect(this, SIGNAL(showInfo(int)), this, SLOT(onShowInfo(int)));
}

Propello::~Propello()
{
    portaShutdown();
}

// create action groups so menus can be checkable
void Propello::setupControls()
{
    QActionGroup* agMain = new QActionGroup(this);
    agMain->addAction(mMain2);
    agMain->addAction(mMain3);
    agMain->addAction(mMain4);

    QActionGroup* agPCI = new QActionGroup(this);
    agPCI->addAction(mPCI2);
    agPCI->addAction(mPCI3);
    agPCI->addAction(mPCI4);

    QActionGroup* agStorage = new QActionGroup(this);
    agStorage->addAction(mStoreAll);
    agStorage->addAction(mStoreForward);
    agStorage->addAction(mStoreBackward);
}

// allow user to change firmware path
void Propello::getFirmwarePath()
{
    bool ok = false;
    QString text = QInputDialog::getText(this, tr("Enter Firmware Path"), tr("Firmware Path:"), QLineEdit::Normal, m_firmwarePath, &ok);

    if (ok && !text.isEmpty())
    {
        m_firmwarePath = text;
    }
}

// allow user to change settings path
void Propello::getSettingsPath()
{
    bool ok = false;
    QString text = QInputDialog::getText(this, tr("Enter Settings Path"), tr("Settings Path:"), QLineEdit::Normal, m_settingsPath, &ok);

    if (ok && !text.isEmpty())
    {
        m_settingsPath = text;
    }
}

// allow user to change cine size
void Propello::setCineSize()
{
    bool ok = false;
    int cine = QInputDialog::getInt(this, tr("Enter Cine Size"), tr("Cine Size (MB):"), m_cineSize, 32, 512, 1, &ok);

    if (ok)
    {
        m_cineSize = cine;
    }
}

// display the system ID for licensing
void Propello::showSystemID()
{
    char sysid[255] = "";
    portaGetSystemId(sysid, 255);

    QMessageBox mb;
    mb.setWindowTitle(tr("System ID"));
    mb.setText((QString("%1").arg(sysid)));
    mb.exec();
}

int newAcqInterrupt(void* param, unsigned char* addr, int blockIndex, int header)
{
    ((Propello*)param)->processRawFrame(addr, blockIndex, header);
    return 0;
}

void Propello::processRawFrame(unsigned char*, int, int header)
{
    if (header && m_probeInfo.motorized && m_probeInfo.motorHomeSensor)
    {
        showInfo(header);
    }
}

void Propello::onShowInfo(int header)
{
    int extraInfo = (header & ~0x0000FFFF) >> 16;
    int endStop = ((extraInfo & 0x00000001) > 0 ? 1 : 0);
    int ccwBar = ((extraInfo & 0x00000002) > 0 ? 1 : 0);
    int stepCount = (extraInfo & 0x0000FFFC);

    wStatus->showMessage(QString(tr("Motor EndStop: %1, CCWBar: %2, Stepcount: %3")).arg(endStop).arg(ccwBar).arg(stepCount));
}

// initialize the ultrasound engine, populate probes, load lists
void Propello::initHardware()
{
    int usm = mMain2->isChecked() ? 2 : (mMain3->isChecked() ? 3 : 4);
    int pci = mPCI2->isChecked() ? 2 : (mPCI3->isChecked() ? 3 : 4);
    int channels = (usm == 2) ? 32 : 64;

    if (!portaInit(m_cineSize * 1024 * 1024, m_firmwarePath.toAscii(), m_settingsPath.toAscii(), "D:/Ultrasonix Settings/", "D:/LUTS/",
            usm, pci, 0, 1, channels))
    {
        QMessageBox mb;
        mb.setText(tr("Propello could not initialize ultrasound"));
        mb.exec();
    }
    else
    {
        loadParams();
        wStatus->showMessage(tr("Successfully initialized ultrasound"), MSG_TIMEOUT);
        wDetect->setEnabled(true);
    }

    //uncheck the probe selection button
    wDetect->setChecked(false);

    portaSetRawDataCallback(newAcqInterrupt, (void*)this);
}

// start imaging
void Propello::runImage()
{
    if (portaIsConnected() && portaRunImage())
    {
        double vps = (double)portaGetFrameRate() / (double)portaGetParam(prmMotorFrames);
        wStatus->showMessage(QString(tr("Imaging Running @ FPS: %1Hz, VPS: %2Hz")).arg(portaGetFrameRate()).arg(vps));
    }
}

// stop imaging
void Propello::stopImage()
{
    if (portaIsConnected() && portaStopImage())
    {
        int disp = 0, vols = portaGetDisplayFrameCount(disp) / portaGetParam(prmMotorFrames);

        wStatus->showMessage(QString(tr("Imaging Stopped. Acquired %1 frames, %2 volumes.")).arg(portaGetDisplayFrameCount(disp)).arg(
                vols));
        wCine->setRange(0, portaGetFrameCount(0));
        wCine->setValue(wCine->maximum());
    }
}

// detect a motorized probe
void Propello::onDetect()
{
    int code;
    char name[80];

    if (m_porta && portaIsConnected())
    {
        code = (char)portaGetProbeID(0);

        // select the code read, and see if it is motorized
        if (portaSelectProbe(code) && portaGetProbeInfo(m_probeInfo) && m_probeInfo.motorized)
        {
            if (portaGetProbeName(name, 80, code))
            {
                wDetect->setText(name);
                wProbeAttributes->setEnabled(true);
            }

            // the 3D/4D probe is always connected to port 0
            portaActivateProbeConnector(0);

            // find and load master preset
            if (loadMasterPreset())
            {
                wCaptureMethod->setEnabled(true);
                wDataSelect->setEnabled(true);
                onCaptureChange(0);
                onDataSelection(0);
            }
        }
        // if probe is motorless
        else
        {
            wDetect->setText(tr("No Motor!"));
        }
    }
}

// load master for current probe
bool Propello::loadMasterPreset()
{
    int i, probe1 = -1, probe2 = -1, probe3 = -1;
    QDir dir;
    QString file, master, path = SETTINGS_PATH;
    path.append("presets/imaging/");
    dir.setPath(path);

    QStringList filters;
    filters << "*.xml";
    dir.setNameFilters(filters);

    QStringList files = dir.entryList(QDir::Files);

    for (i = 0; i < files.size(); i++)
    {
        file = dir.path() + "/" + files[i];

        portaGetPresetProbeID(file.toAscii(), probe1, probe2, probe3);
        if (probe1 == portaGetCurrentProbeID() || probe2 == portaGetCurrentProbeID() || probe3 == portaGetCurrentProbeID())
        {
            if (portaIsMasterPreset(file.toAscii()))
            {
                master = file;
                break;
            }
        }
    }

    if (master.length())
    {
        if (portaLoadPreset(master.toAscii()))
        {
            loadParams();
            return loadMode(BMode, 0);
        }
    }

    return false;
}

// user changed capture method (auto vs manual)
void Propello::onCaptureChange(int index)
{
    bool running = portaIsImaging() != 0;

    // ensure we are stopped
    if (running)
    {
        portaStopImage();
    }

    portaSetParamI(prmMotorStatus, index == 0 ? 1 : 0);
    wStoreVolumes->setEnabled(true);
    wHomeMotor->setEnabled(index == 1);

    if (running)
    {
        portaRunImage();
    }
}

// user selected a new mode
void Propello::onDataSelection(int index)
{
    imagingMode mode = BMode;

    if (index >= 0 && index < 3)
    {
        mode = BMode;
    }
    else if (index == 3)
    {
        mode = MMode;
    }
    else if (index == 4)
    {
        mode = ColourMode;
    }

    loadMode(mode, 0);
}

// initialize the imaging mode
bool Propello::loadMode(int mode, int)
{
    // create mode
    if (!portaInitMode((imagingMode)mode))
    {
        QMessageBox mb;
        mb.setText(tr("Porta could not initialize the imaging mode"));
        mb.exec();
        return false;
    }

    wBImage->init(0);

    if (mode == ColourMode)
    {
        wBImage->loadColorMap(":/res/map.bmp");
    }

    refreshParams();

    return true;
}

// show probe details
void Propello::onProbeAttributes()
{
    QString msg;
    probeInfo nfo;

    if (m_porta && portaGetProbeInfo(nfo))
    {
        msg = QString("Base(Radius:%1, Elems: %2, Pitch: %3m) Motor(FOV: %4, Radius: %5, Steps: %6, Deg/Step: %7)")
            .arg(nfo.radius / 1000.0).arg(nfo.elements).arg(nfo.pitch).arg(nfo.motorFov / 1000.0).arg(nfo.motorRadius / 1000.0).arg(
                nfo.motorSteps).arg((double)nfo.motorFov / (double)nfo.motorSteps / 1000.0);

        wStatus->showMessage(msg);
    }
}

// increment parameter
void Propello::incParamMotor()
{
    QString prm;
    int max = 0;
    probeInfo nfo;

    if (m_porta)
    {
        portaGetProbeInfo(nfo);
        max = nfo.motorFov / 1000;
    }

    if (!wParams->currentItem())
    {
        return;
    }

    switch (wParams->currentItem()->row())
    {
    case 1: prm = prmMotorSteps;
        break;
    case 2: prm = prmMotorFrames;
        break;
    case 3: prm = prmMotorFOV;
        break;
    case 5: m_motorstart = (m_motorstart < max) ? m_motorstart + 1 : max;
        break;
    case 6: stepMotor(true);
        break;
    }

    if (!prm.isEmpty())
    {
        portaCycleParam(prm.toAscii().constData(), true);
    }

    refreshParams();
}

// decrement parameter
void Propello::decParamMotor()
{
    QString prm;

    if (!wParams->currentItem())
    {
        return;
    }

    switch (wParams->currentItem()->row())
    {
    case 1: prm = prmMotorSteps;
        break;
    case 2: prm = prmMotorFrames;
        break;
    case 3: prm = prmMotorFOV;
        break;
    case 5: m_motorstart = (m_motorstart > 0) ? m_motorstart - 1 : 0;
        break;
    case 6: stepMotor(false);
        break;
    }

    if (!prm.isEmpty())
    {
        portaCycleParam(prm.toAscii().constData(), false);
    }

    refreshParams();
}

// applies parameter when user has edited (currently motor home only)
void Propello::onApply()
{
    m_motorstart = wParams->item(5, 0)->text().toInt();

    // if in manual mode and motor start option is highlighted
    // move the motor to start position and update the motor current location
    if (wParams->currentItem()->row() == 5 && (wCaptureMethod->currentIndex() == 1))
    {
        m_motorpos = m_motorstart;
        onHomeMotor();
    }
}

// homes the motor
void Propello::onHomeMotor()
{
    bool running = portaIsImaging() != 0;

    if (running)
    {
        portaStopImage();
    }

    m_motorpos = portaGoToPosition(m_motorstart);

    if (running)
    {
        portaRunImage();
    }

    refreshParams();
}

// steps the motor manually
void Propello::stepMotor(bool fwd)
{
    if (wCaptureMethod->currentIndex() == 1)
    {
        bool running = portaIsImaging() != 0;
        double stepamt;

        // use the step amount from the automatic acquisition
        int steps = portaGetParam(prmMotorSteps);

        if (running)
        {
            portaStopImage();
        }

        stepamt = portaStepMotor(!fwd, steps);

        if (running)
        {
            portaRunImage();
        }

        m_motorpos += stepamt;
    }
}

// refreshes table parameters
void Propello::refreshParams()
{
    probeInfo nfo;
    portaGetProbeInfo(nfo);

    int steps = portaGetParam(prmMotorSteps);
    double degPerStep = (double)nfo.motorFov / (double)nfo.motorSteps / 1000.0;

    wParams->item(1, 0)->setText(QString("%1").arg((double)steps * degPerStep));
    wParams->item(2, 0)->setText(QString("%1").arg(portaGetParam(prmMotorFrames)));
    wParams->item(3, 0)->setText(QString("%1").arg((double)portaGetParam(prmMotorFrames) * degPerStep * (double)steps));
    wParams->item(5, 0)->setText(QString("%1").arg(m_motorstart));
    wParams->item(6, 0)->setText(QString("%1").arg(m_motorpos));

    if (!wParams2->currentItem())
    {
        return;
    }

    QString prm;
    for (int i = 0; i< wParams2->rowCount(); i++)
    {
        prm = wParams2->item(i, 1)->data(Qt::UserRole).toString().trimmed();
        wParams2->item(i, 1)->setText(getParamStringValue(prm));
    }
}

// user scrolled cine
void Propello::cineScroll(int value)
{
    if (!portaIsImaging())
    {
        portaProcessCineImage(0, value);
    }
}

// store post-scan converted B volumes
void Propello::onStoreVolume()
{
    SaveDlg dlg(((wCaptureMethod->currentIndex() == 0) ? storeAutoVolume : storeManualVolume));
    dlg.exec();
}

// fill modes and parameters lists
void Propello::loadParams()
{
    int i, count;
    QTableWidgetItem* item;

    wParams2->blockSignals(true);

    count = portaGetNumParams();
    bool ret;
    char prm[100];
    wParams2->setRowCount(count);

    for (i = 0; i < count; i++)
    {
        ret = portaGetListParam(prm, 100, i) != 0;
        if (ret)
        {
            item = new QTableWidgetItem(prm);
            item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            item->setData(Qt::UserRole, prm);
            wParams2->setItem(i, 0, item);

            item = new QTableWidgetItem(getParamStringValue(prm));
            item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable | Qt::ItemIsEditable);
            item->setData(Qt::UserRole, prm);
            wParams2->setItem(i, 1, item);
        }
    }
    wParams2->blockSignals(true);
}

QString Propello::getParamStringValue(QString prm)
{
    prm = prm.trimmed();
    portaVarType type = pVariableUnknown;

    if (portaGetParamType(prm.toAscii(), type))
    {
        char val[100];
        int intVal;
        portaGainCurve gVal;
        portaCurve cVal;
        portaRect rVal;
        portaPoint pVal;
        std::vector< int > arrayVal(200);
        int arrayValCount = arrayVal.size();

        switch (type)
        {
        case pInteger:
        {
            if (portaGetParamI(prm.toAscii(), intVal))
            {
                return QString("%1").arg(intVal);
            }
        }
        break;
        case pPoint:
        {
            if (portaGetParamP(prm.toAscii(), pVal))
            {
                return QString("[P] %1,%2").arg(pVal.x).arg(pVal.y);
            }
        }
        break;
        case pString:
        {
            if (portaGetParamS(prm.toAscii(), val, 100))
            {
                return QString("%1").arg(val);
            }
        }
        break;
        case pGainCurve:
        {
            if (portaGetParamGC(prm.toAscii(), gVal))
            {
                return QString("[C] %1,%2,%3,%4,%5,%6,%7,%8").arg(gVal.val[0]).arg(gVal.val[1]).arg(gVal.val[2]).arg(gVal.val[3])
                       .arg(gVal.val[4]).arg(gVal.val[5]).arg(gVal.val[6]).arg(gVal.val[7]);
            }
        }
        break;
        case pRectangle:
        {
            if (portaGetParamR(prm.toAscii(), rVal))
            {
                return QString("[R] %1,%2,%3,%4").arg(rVal.left).arg(rVal.top).arg(
                    rVal.right).arg(rVal.bottom);
            }
        }
        break;
        case pCurve:
        {
            if (portaGetParamC(prm.toAscii(), cVal))
            {
                return QString("[C] %1,%2,%3,%4").arg(cVal.t).arg(cVal.m).arg(cVal.b).arg(cVal.vm);
            }
        }
        break;
        case pArray:
        {
            if (portaGetParamA(prm.toAscii(), &arrayVal.front(), &arrayValCount))
            {
                QString value;
                for (int i = 0; i < arrayValCount; i++)
                {
                    value.append(QString("%1,").arg(arrayVal.at(i)));
                }
                return value;
            }
        }
        break;
        }
    }

    return "n/a";
}

// increment parameter by internal step
void Propello::incParam()
{
    if (!wParams2->currentItem())
    {
        return;
    }

    QString prm = wParams2->currentItem()->data(Qt::UserRole).toString();
    if (portaCycleParam(prm.trimmed().toAscii(), true))
    {
        refreshParams();
    }
}

// decrement parameter by internal step
void Propello::decParam()
{
    if (!wParams2->currentItem())
    {
        return;
    }

    QString prm = wParams2->currentItem()->data(Qt::UserRole).toString();
    if (portaCycleParam(prm.trimmed().toAscii(), false))
    {
        refreshParams();
    }
}
