%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Loads the ultrasound volume data saved from the Propello application
%%
%% Inputs:  
%%     datapath - The path of the data to open
%%     volN   - The volume to retrieve
%%
%% Return:
%%     volume -  The volume data returned into a 3D array
%%   header:
%%      header.datatype -   Data type (0 = prescan b, 1  = postscan b, 2 = rf)
%%      header.numVols -    The number of volumes in the file
%%      header.fpV -        Frames per volume
%%      header.h -          Image height of the image (samples per line)
%%      header.w -          Image width (scan lines per frame)
%%      header.ss -         Sample size
%%      degPerFr -          Degrees Per Frame
%%
%% Corina Leung, corina.leung@ultrasonix.com
%% Ultrasonix Medical Corporation Nov 2007
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [volume, header]  = loadvol(datapath, volN)

if nargin == 1
    volN = 1;
end

fid= fopen(datapath, 'r');

if( fid == -1)
    error('Cannot open file');
end

% read the header info
volinfo = fread(fid, 7, 'int');
header = struct('datatype', 0, 'numVols', 0, 'fpV', 0, 'w', 0, 'h', 0, 'ss', [0,0], 'degPerFr', [0,0]);
header.datatype = volinfo(1);  
header.numVols = volinfo(2);
header.fpV = volinfo(3);
header.w = volinfo(4);
header.h = volinfo(5);
header.ss = volinfo(6);
header.degPerFr = volinfo(7);

%buffer = zeros(1,volN*(h*w*fpV));
volume = [];

if(header.datatype == 0 || header.datatype == 1)
  % read volume data
  volData = fread(fid, inf, 'uchar=>uchar');
  
    if(header.numVols < volN)
        error('Error invalid volume number');
        volume = 0;
        return
    end

    buffer = volData((volN-1)*(header.h*header.w*header.fpV)+1 : volN*(header.h*header.w*header.fpV));

    % prescan b 
    if(header.datatype == 0)
        for i = 1 : header.fpV
            volume(:,:,i) = reshape(buffer((i-1)*header.h*header.w+1 : i*header.h*header.w), header.h, header.w);  
        end
    end
    
    % postscan b
    if(header.datatype == 1)
        for i = 1 : header.fpV
            temp = reshape(buffer((i-1)*header.h*header.w+1 : i*header.h*header.w), header.w, header.h);  
            volume(:,:,i) = imrotate(temp, -90);
        end
    end

% rf data
else
    other = fread(fid, (volN-1)*(header.h*header.w*header.fpV),'int16');
    for i = 1:header.fpV
        %the actual data...
        [temp,count] = fread(fid,[header.h header.w],'int16'); 
        volume(:,:,i) = int16(temp);
    end
    
end
    
fclose(fid);

