#include "stdafx.h"
#include "PortaBDisplay.h"
#define UMC_FRAME_ID_MASK (0x0000FFFF)

PortaBDisplay::PortaBDisplay(QWidget* parent) : PortaDisplay(parent)
{
}

PortaBDisplay::~PortaBDisplay()
{
}

// loads a color map for doppler display
bool PortaBDisplay::loadColorMap(const QString file)
{
    QImage img;
    if (!img.load(file))
    {
        return false;
    }

    return portaImportColorMap(0, (unsigned int*)img.constBits()) != 0;
}

// sets the image data for the main display
bool PortaBDisplay::setImgData(int header)
{
    bool ret = false;

    if (header)
    {
        setDisplayHeader(header);
    }

    if (!m_image || m_index < 0)
    {
        return false;
    }

    unsigned char* buffer = m_image->bits();
    int mode = portaGetCurrentMode();

    if (mode == ColourMode || mode == DualColourMode || mode == TriplexMode)
    {
        ret = portaGetColorImage(m_index, buffer) != 0;

        // To retrieve B mode and color data separately make the following
        // two calls to store the data into separate buffers
        // portaGetBwImage(m_index, buffer1, true);
        // portaGetColorImageOnly(m_index, buffer2);
    }
    else
    {
        ret = portaGetBwImage(m_index, buffer, true) != 0;
    }

    if (ret)
    {
        scene()->invalidate();
    }

    return true;
}

void PortaBDisplay::setDisplayHeader(int header)
{
    probeInfo nfo;
    portaGetProbeInfo(nfo);
    // This part is for the TEE probes only
    if (nfo.supportsTempAngleSensor)
    {
        // header from display is the TEE data info header
        short teeDataHeader = static_cast<short>(header);
        bool errorStatus = (teeDataHeader & 0x00002000) ? true : false;
        int teeT_P = (teeDataHeader & 0x00001000);
        int m_tnTEETemperature, m_tnTEEPhase;
        if (teeDataHeader && !errorStatus)
        {
            if (teeT_P)   // if T/P = 1 temperature, = 0 phase
            {
                m_tnTEETemperature = (teeDataHeader & 0x00000FFF);
            }
            else
            {
                m_tnTEEPhase = (teeDataHeader & 0x00000FFF);
                m_tnTEEPhase = m_tnTEEPhase;
            }
        }
    }
}

// draws the foreground graphics
void PortaBDisplay::drawForeground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawBackground(painter, r);

    drawFocus(painter);
    drawDepth(painter);

    if (m_mode == ColourMode || m_mode == TriplexMode || m_mode == DualColourMode)
    {
        drawColorBox(painter);
    }

    if (m_mode == PwMode || m_mode == TriplexMode)
    {
        drawGate(painter);
    }

    if (m_mode == MMode)
    {
        drawMWindow(painter);
    }
}

// draws the focus markers
void PortaBDisplay::drawFocus(QPainter* painter)
{
    int i, start, fc, fd, fs, mx, my, pels;

    if ((fc = portaGetParam(prmBFocusCount)) == -1)
    {
        fc = 1;
    }

    if ((fd = portaGetParam(prmBFocusDepth)) == -1)
    {
        return;
    }

    if ((fs = portaGetParam(prmBFocusSpacing)) == -1)
    {
        return;
    }

    // if odd # of focus
    if (fc % 2)
    {
        start = fd - (((fc - 1) / 2) * fs);
    }
    // if even # of focus
    else
    {
        start = (fd - (fs / 2)) - (fs * ((fc / 2) - 1));
    }

    portaGetMicronsPerPixel(m_index, mx, my);

    painter->setPen(Qt::white);

    for (i = 0; i < fc; i++)
    {
        pels = (start + (i * fs)) / my;
        painter->drawEllipse(5, pels, 4, 4);
    }
}

// draws the depth scale
void PortaBDisplay::drawDepth(QPainter* painter)
{
    int i, mx, my, dp, pels;
    QPainterPath pp;

    dp = portaGetParam(prmBImageDepth) / 10;

    portaGetMicronsPerPixel(m_index, mx, my);

    for (i = 0; i <= dp; i++)
    {
        if (my > 0)
        {
            pels = (10000 * i) / my;
            pp.moveTo(0, pels);
            pp.lineTo((i % 5) ? 3 : 8, pels);
        }
    }

    painter->setPen(Qt::white);
    painter->drawPath(pp);
}

// draw PW/CW gate
void PortaBDisplay::drawGate(QPainter* painter)
{
    int dp, pos, sz, sf, x, y, steer;
    int vsound = 1540;

    dp = portaGetParam(prmPwGateDepth);
    pos = portaGetParam(prmPwGatePos);
    sz = portaGetParam(prmPwGateSize);
    sf = portaGetParam(prmBSamplingFreq);
    steer = portaGetParam(prmPWSteer);

    QPainterPath pp;

    if (portaGetPixelCoordinates(m_index, pos, micronsToUSSamples(dp, sf, vsound), x, y, false, steer))
    {
        pp.moveTo(x - 5, y);
        pp.lineTo(x + 5, y);
    }

    if (portaGetPixelCoordinates(m_index, pos, micronsToUSSamples(dp + sz, sf, vsound), x, y, false, steer))
    {
        pp.moveTo(x - 5, y);
        pp.lineTo(x + 5, y);
    }

    painter->setPen(OVERLAYCOLOR);
    painter->drawPath(pp);
}

// draw M window
void PortaBDisplay::drawMWindow(QPainter* painter)
{
    int dp, pos, sz, steer, sf, x, y;
    int vsound = 1540;

    dp = portaGetParam(prmMDepth);
    pos = portaGetParam(prmMPos);
    sz = portaGetParam(prmMZoom);
    sf = portaGetParam(prmBSamplingFreq);
    steer = portaGetParam(prmMSteer);

    QPainterPath pp;

    if (portaGetPixelCoordinates(m_index, pos, micronsToUSSamples(dp, sf, vsound), x, y, false, steer))
    {
        pp.moveTo(x - 5, y);
        pp.lineTo(x + 5, y);
    }

    if (portaGetPixelCoordinates(m_index, pos, micronsToUSSamples(dp + sz, sf, vsound), x, y, false, steer))
    {
        pp.moveTo(x - 5, y);
        pp.lineTo(x + 5, y);
    }

    painter->setPen(OVERLAYCOLOR);
    painter->drawPath(pp);
}

// draw color box
void PortaBDisplay::drawColorBox(QPainter* painter)
{
    int x[4], y[4];
    portaRect arcRect;

    QPainterPath pp;

    if (portaGetColorBox(m_index, x, y))
    {
        pp.moveTo(x[0], y[0]);
        pp.lineTo(x[3], y[3]);

        pp.moveTo(x[1], y[1]);
        pp.lineTo(x[2], y[2]);

        // draw the horizontal lines
        if (portaGetHorizontalArcRect(m_index, arcRect, true, true))
        {
        }
        else
        {
            pp.moveTo(x[0], y[0]);
            pp.lineTo(x[1], y[1]);
        }

        if (portaGetHorizontalArcRect(m_index, arcRect, true, false))
        {
        }
        else
        {
            pp.moveTo(x[3], y[3]);
            pp.lineTo(x[2], y[2]);
        }

        painter->setPen(OVERLAYCOLOR);
        painter->drawPath(pp);
    }
}

void PortaBDisplay::mouseReleaseEvent(QMouseEvent* evt)
{
    int vsound = 1540;

    if (m_mode == ColourMode || m_mode == DualColourMode || m_mode == TriplexMode)
    {
        portaRect r;
        int x, y;
        bool changeColorBoxSize = false;

        if (changeColorBoxSize)
        {
            // get the color box in US coordinates
            if (portaGetParamR("color-color rect", r))
            {
                // convert the mouse cursor into an ultrasound coordinate to
                // be used for the bottom right of the color box
                if (portaGetUltrasoundCoordinates(m_index, evt->x(), evt->y(), x, y, true))
                {
                    // set the bottom right for the new color box
                    r.right = x;
                    r.bottom = y;

                    while (r.bottom % 4)
                    {
                        r.bottom++;
                    }

                    portaSetParamR("color-color rect", r);
                }
            }
        }
        else
        {
            // get the new color box by using the cursor position as the center of the color box
            if (portaGetNewColorBox(m_index, evt->x(), evt->y(), r, true))
            {
                portaSetParamR(prmColorBox, r);
            }
        }
    }
    else if (m_mode == PwMode)
    {
        int x, y;
        int sf = portaGetParam(prmBSamplingFreq);
        int steer = portaGetParam(prmPWSteer);

        if (portaGetUltrasoundCoordinates(m_index, evt->x(), evt->y(), x, y, false, steer))
        {
            portaSetParamI(prmPwGateDepth, usSamplesToMicrons(y, sf, vsound));
            portaSetParamI(prmPwGatePos, x);
        }
    }
    else if (m_mode == MMode)
    {
        int x, y;
        int sf = portaGetParam(prmBSamplingFreq);
        int steer = portaGetParam(prmMSteer);

        if (portaGetUltrasoundCoordinates(m_index, evt->x(), evt->y(), x, y, false, steer))
        {
            portaSetParamI(prmMDepth, usSamplesToMicrons(y, sf, vsound));
            portaSetParamI(prmMPos, x);
        }
    }

    PortaDisplay::mouseReleaseEvent(evt);
}
