#include "stdafx.h"
#include "PortaDemo.h"

#ifndef FIRMWARE_PATH
    #define FIRMWARE_PATH "../../fw/"
#endif

#ifndef SETTINGS_PATH
    #define SETTINGS_PATH "../../dat/"
#endif

#ifndef LICENSE_PATH
    #define LICENSE_PATH ("D:/Ultrasonix Settings/")
#endif

#define PRESETS_PATH      (SETTINGS_PATH "presets/imaging/")

#define MSG_TIMEOUT       3000 // ms
#define CINE_SIZE         128 // MB
#define UMC_FRAME_ID_MASK (0x0000FFFF)

namespace
{
    QString appendPathSeparator(const QString& s)
    {
        return s.right(1) == "/" ? s : s + "/";
    }
}

PortaDemo::PortaDemo(QWidget* parent) : QMainWindow(parent)
{
    setupUi(this);
    setupControls();

    m_firmwarePath = FIRMWARE_PATH;
    m_settingsPath = SETTINGS_PATH;
    m_licensePath = LICENSE_PATH;
    m_cineSize = CINE_SIZE;
    m_tnTEETemperature = -1.0;
    m_tnTEEPhase = -1.0;

    connect(this, SIGNAL(headerInfo(int)), this, SLOT(onHeaderInfo(int)));
}

PortaDemo::~PortaDemo()
{
    portaShutdown();
}

int newAcqInterrupt(void* param, unsigned char* addr, int blockIndex, int header)
{
    // extract the frame header information
    ((PortaDemo*)param)->processRawFrame(addr, blockIndex, header);
    return 0;
}

void PortaDemo::processRawFrame(unsigned char*, int, int header)
{
    if (header)
    {
        headerInfo(header);
    }
}

// create action groups so menus can be checkable
void PortaDemo::setupControls()
{
    int i;

    QActionGroup* agMain = new QActionGroup(this);
    agMain->addAction(mMain2);
    agMain->addAction(mMain3);
    agMain->addAction(mMain4);

    QActionGroup* agPCI = new QActionGroup(this);
    agPCI->addAction(mPCI2);
    agPCI->addAction(mPCI3);
    agPCI->addAction(mPCI4);

    wModes->blockSignals(true);
    for (i = 0; i < wModes->count(); i++)
    {
        wModes->setItemData(i, BMode + i);
    }
    wModes->setCurrentIndex(0);
    wModes->blockSignals(false);
}

// fill modes and parameters lists
void PortaDemo::loadParams()
{
    int i, count;
    QTableWidgetItem* item;

    wParams->blockSignals(true);

    count = portaGetNumParams();
    bool ret;
    char prm[100];
    wParams->setRowCount(count);
    
    //disable sorting before setting an item otherwise it will mess up the parameters list 
    wParams->setSortingEnabled(false);

    for (i = 0; i < count; i++)
    {
        ret = portaGetListParam(prm, 100, i) != 0;
        if (ret)
        {
            item = new QTableWidgetItem(prm);
            item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable);
            item->setData(Qt::UserRole, prm);
            wParams->setItem(i, 0, item);

            item = new QTableWidgetItem(getParamStringValue(prm));
            item->setFlags(Qt::ItemIsEnabled | Qt::ItemIsSelectable | Qt::ItemIsEditable);
            item->setData(Qt::UserRole, prm);
            wParams->setItem(i, 1, item);
        }
    }

    wParams->setSortingEnabled(true);
    wParams->blockSignals(true);
}

// allow user to change firmware path
void PortaDemo::getFirmwarePath()
{
    QString path = QFileDialog::getExistingDirectory(this, tr("Select Firmware Path"), m_firmwarePath);
    if (!path.isEmpty())
    {
        m_firmwarePath = QDir::fromNativeSeparators(path);
    }
}

// allow user to change settings path
void PortaDemo::getSettingsPath()
{
    QString path = QFileDialog::getExistingDirectory(this, tr("Select Settings Path"), m_settingsPath);
    if (!path.isEmpty())
    {
        m_settingsPath = QDir::fromNativeSeparators(path);
    }
}

// allow user to change cine size
void PortaDemo::setCineSize()
{
    bool ok = false;
    int cine = QInputDialog::getInt(this, tr("Enter Cine Size"), tr("Cine Size (MB):"), m_cineSize, 32, 512, 1, &ok);

    if (ok)
    {
        m_cineSize = cine;
    }
}

// display the system ID for licensing
void PortaDemo::showSystemID()
{
    char sysid[255] = "";
    portaGetSystemId(sysid, 255);

    QMessageBox mb;
    mb.setWindowTitle(tr("System ID"));
    mb.setText((QString("%1").arg(sysid)));
    mb.exec();
}

// initialize the ultrasound engine, populate probes, load lists
void PortaDemo::initHardware()
{
    int usm = mMain2->isChecked() ? 2 : (mMain3->isChecked() ? 3 : 4);
    int pci = mPCI2->isChecked() ? 2 : (mPCI3->isChecked() ? 3 : 4);
    int channels = (usm == 2) ? 32 : 64;

    m_firmwarePath = appendPathSeparator(m_firmwarePath);
    m_settingsPath = appendPathSeparator(m_settingsPath);
    m_licensePath = appendPathSeparator(m_licensePath);
    if (!portaInit(m_cineSize * 1024 * 1024, m_firmwarePath.trimmed().toAscii(), m_settingsPath.trimmed().toAscii(),
            m_licensePath.trimmed().toAscii(), "D:/LUTS/",
            usm, pci, 0, 1, channels))
    {
        QMessageBox mb;
        mb.setText(tr("Porta could not initialize ultrasound"));
        mb.exec();
    }
    else
    {
        readProbes();
        loadParams();
        wStatus->showMessage(tr("Successfully intitialized ultrasound"), MSG_TIMEOUT);
    }

    // reset the connector buttons
    QFont oldFont("Verdana", 8);
    wProbe1->setFont(oldFont);
    wProbe2->setFont(oldFont);
    wProbe3->setFont(oldFont);

    // Use the below function to set a transmit watchdog
    // On calling runImage() the watchdog will turn off transmit after 30 seconds of imaging
    // To reset, stop image and run image again
    // portaSetWatchdog(30000);

    portaSetRawDataCallback(newAcqInterrupt, (void*)this);
}

// start imaging
void PortaDemo::runImage()
{
    if (portaIsConnected() && portaRunImage())
    {
        wStatus->showMessage(tr(QString("Imaging Running @ %1Hz").arg(portaGetFrameRate()).trimmed().toAscii()));
    }
}

// stop imaging
void PortaDemo::stopImage()
{
    if (portaIsImaging() && portaStopImage())
    {
        wStatus->showMessage(tr("Imaging Stopped"));
        wCine->setRange(0, portaGetFrameCount(0));
        wCine->setValue(wCine->maximum());
    }
}

// select first probe
void PortaDemo::selectProbe1()
{
    portaSelectProbe(portaGetProbeID(0));
    portaActivateProbeConnector(0);
    loadPresets();

    wStatus->showMessage(tr("Selected first probe connector"), MSG_TIMEOUT);

    QFont sel(font().family(), font().pointSize(), QFont::Bold);
    QFont f(font().family(), font().pointSize());
    wProbe1->setFont(sel);
    wProbe2->setFont(f);
    wProbe3->setFont(f);
}

// select second probe
void PortaDemo::selectProbe2()
{
    portaSelectProbe(portaGetProbeID(1));
    portaActivateProbeConnector(1);
    loadPresets();

    wStatus->showMessage(tr("Selected second probe connector"), MSG_TIMEOUT);

    QFont sel(font().family(), font().pointSize(), QFont::Bold);
    QFont f(font().family(), font().pointSize());
    wProbe1->setFont(f);
    wProbe2->setFont(sel);
    wProbe3->setFont(f);
}

// select third probe
void PortaDemo::selectProbe3()
{
    portaSelectProbe(portaGetProbeID(2));
    portaActivateProbeConnector(2);
    loadPresets();

    wStatus->showMessage(tr("Selected third probe connector"), MSG_TIMEOUT);

    QFont sel(font().family(), font().pointSize(), QFont::Bold);
    QFont f(font().family(), font().pointSize());
    wProbe1->setFont(f);
    wProbe2->setFont(f);
    wProbe3->setFont(sel);
}

// user selected a preset from the combobox
void PortaDemo::selectPreset(QString preset)
{
    preset.insert(0, PRESETS_PATH);

    if (portaLoadPreset(preset.trimmed().toAscii()))
    {
        loadParams();
        initMode();
        wStatus->showMessage(tr("Successfully loaded preset"), MSG_TIMEOUT);
    }
}

// user wants to save a preset
void PortaDemo::savePreset()
{
    bool ok;
    QString preset = "MyPreset";
    preset = QInputDialog::getText(this, tr("Save Preset"), tr("Preset Name:"), QLineEdit::Normal, preset, &ok);

    if (ok && !preset.isEmpty())
    {
        if (portaSavePreset(preset.trimmed().toAscii(), false))
        {
            wStatus->showMessage(tr("Successfully saved preset"), MSG_TIMEOUT);
        }
    }
}

// read probes and setup controls
void PortaDemo::readProbes()
{
    int i, code[3];
    char name[80];
    QAbstractButton* wProbe = 0;

    for (i = 0; i < 3; i++)
    {
        code[i] = portaGetProbeID(i);
    }

    for (i = 0; i < 3; i++)
    {
        wProbe = (i == 0) ? wProbe1 : ((i == 1) ? wProbe2 : wProbe3);

        if (portaGetProbeName(name, 80, code[i]))
        {
            wProbe->setText(name);
            wProbe->setEnabled(true);
        }
        else
        {
            wProbe->setText("");
            wProbe->setEnabled(false);
        }
    }
}

// load presets for current probe
void PortaDemo::loadPresets()
{
    wPresets->clear();

    int i, probe1 = -1, probe2 = -1, probe3 = -1;
    QDir dir;
    QString file, master, path = m_settingsPath;
    path.append("presets/imaging/");
    dir.setPath(path);

    QStringList filters;
    filters << "*.xml";
    dir.setNameFilters(filters);

    QStringList files = dir.entryList(QDir::Files);

    for (i = 0; i < files.size(); i++)
    {
        file = dir.path() + "/" + files[i];
        probe1 = -1;
        probe2 = -1;
        probe3 = -1;
        portaGetPresetProbeID(file.trimmed().toAscii(), probe1, probe2, probe3);
        if ((probe1 == portaGetCurrentProbeID() &&
             probe1 != -1) ||
            (probe2 == portaGetCurrentProbeID() && probe1 != -1) || (probe3 == portaGetCurrentProbeID() && probe1 != -1))
        {
            wPresets->addItem(files[i]);

            if (portaIsMasterPreset(file.trimmed().toAscii()))
            {
                master = file;
                wPresets->setCurrentIndex(wPresets->findText(files[i]));
            }
        }
    }

    if (master.length())
    {
        if (portaLoadPreset(master.trimmed().toAscii()))
        {
            loadParams();
            initMode();
        }
    }
}

// user changed mode from combobox
void PortaDemo::changeMode(int)
{
    if (initMode())
    {
        // check if we're in color mode
        wBImage->loadColorMap(":/res/map.bmp");
        wStatus->showMessage(tr("Successfully switched modes"), MSG_TIMEOUT);
    }
}

// initialize the imaging mode
bool PortaDemo::initMode()
{
    int modeName = wModes->itemData(wModes->currentIndex()).toInt();    
    // mode list: 0 = B, 1 = M, 2 = Color, 3 = PW, 4 = Triplex, 5 = Compound
    // imagingMode enumeration 0 = B, 1 = M, 2 = Color, 3 = PW, 4 = Triplex, 8 = Compound

    int mode = (modeName == 5) ? 8 : modeName;

    if (portaGetCurrentProbeID() == -1)
    {
        return false;
    }

    // create mode
    if (!portaInitMode((imagingMode)mode))
    {
        QMessageBox mb;
        mb.setText(tr("Porta could not initialize the imaging mode"));
        mb.exec();
        return false;
    }

    wBImage->init(0, mode);

    if (mode == MMode || mode == PwMode || mode == TriplexMode)
    {
        wSpectrum->init(1, mode);
    }

    return true;
}

// user selected a parameter in the control
void PortaDemo::onSelectParam()
{
    refreshParamValue();
}

// increment parameter by internal step
void PortaDemo::incParam()
{
    if (!wParams->currentItem())
    {
        return;
    }

    QString prm = wParams->currentItem()->data(Qt::UserRole).toString();
    if (portaCycleParam(prm.trimmed().toAscii(), true))
    {
        refreshParamValue();
    }
}

// decrement parameter by internal step
void PortaDemo::decParam()
{
    if (!wParams->currentItem())
    {
        return;
    }

    QString prm = wParams->currentItem()->data(Qt::UserRole).toString();
    if (portaCycleParam(prm.trimmed().toAscii(), false))
    {
        refreshParamValue();
    }
}

QString PortaDemo::getParamStringValue(QString prm)
{
    prm = prm.trimmed();
    portaVarType type = pVariableUnknown;

    if (portaGetParamType(prm.toAscii(), type))
    {
        char val[100];
        int intVal;
        portaGainCurve gVal;
        portaCurve cVal;
        portaRect rVal;
        portaPoint pVal;
        std::vector< int > arrayVal(200);
        int arrayValCount = arrayVal.size();

        switch (type)
        {
        case pInteger:
        {
            if (portaGetParamI(prm.toAscii(), intVal))
            {
                return QString("%1").arg(intVal);
            }
        }
        break;
        case pPoint:
        {
            if (portaGetParamP(prm.toAscii(), pVal))
            {
                return QString("[P] %1,%2").arg(pVal.x).arg(pVal.y);
            }
        }
        break;
        case pString:
        {
            if (portaGetParamS(prm.toAscii(), val, 100))
            {
                return QString("%1").arg(val);
            }
        }
        break;
        case pGainCurve:
        {
            if (portaGetParamGC(prm.toAscii(), gVal))
            {
                return QString("[C] %1,%2,%3,%4,%5,%6,%7,%8").arg(gVal.val[0]).arg(gVal.val[1]).arg(gVal.val[2]).arg(gVal.val[3])
                       .arg(gVal.val[4]).arg(gVal.val[5]).arg(gVal.val[6]).arg(gVal.val[7]);
            }
        }
        break;
        case pRectangle:
        {
            if (portaGetParamR(prm.toAscii(), rVal))
            {
                return QString("[R] %1,%2,%3,%4").arg(rVal.left).arg(rVal.top).arg(
                    rVal.right).arg(rVal.bottom);
            }
        }
        break;
        case pCurve:
        {
            if (portaGetParamC(prm.toAscii(), cVal))
            {
                return QString("[C] %1,%2,%3,%4").arg(cVal.t).arg(cVal.m).arg(cVal.b).arg(cVal.vm);
            }
        }
        break;
        case pArray:
        {
            if (portaGetParamA(prm.toAscii(), &arrayVal.front(), &arrayValCount))
            {
                QString value;
                for (int i = 0; i < arrayValCount; i++)
                {
                    value.append(QString("%1,").arg(arrayVal.at(i)));
                }
                return value;
            }
        }
        break;
        }
    }

    return "n/a";
}

// refreshes the currently selected parameter value
void PortaDemo::refreshParamValue()
{
    if (!wParams->currentItem())
    {
        return;
    }

    QString prm;
    for (int i = 0; i < wParams->rowCount(); i++)
    {
        prm = wParams->item(i, 1)->data(Qt::UserRole).toString().trimmed();
        wParams->item(i, 1)->setText(getParamStringValue(prm));
    }
}

// user scrolled cine
void PortaDemo::cineScroll(int value)
{
    if (!portaIsImaging())
    {
        portaProcessCineImage(0, value);
    }
}

void PortaDemo::onHeaderInfo(int header)
{
    if (header)
    {
        probeInfo nfo;
        portaGetProbeInfo(nfo);
        // This part is for the TEE probes only
        if (nfo.supportsTempAngleSensor)
        {
            int headerInfo = header;
            short frameID = static_cast< short >(headerInfo & UMC_FRAME_ID_MASK);
            // this portion of the header is only available if the probe is a TEE that supports temperature and phase readout
            short teeDataHeader = static_cast< short >((headerInfo & ~UMC_FRAME_ID_MASK) >> 16);

            bool errorStatus = (teeDataHeader & 0x00002000) ? true : false;
            int teeT_P = (teeDataHeader & 0x00001000);

            if (teeDataHeader && !errorStatus)
            {
                if (teeT_P)   // if T/P = 1 temperature, = 0 phase
                {
                    m_tnTEETemperature = RawMiniTeeToTemperature(teeDataHeader & 0x00000FFF);
                }
                else
                {
                    m_tnTEEPhase = RawMiniTnTeeToPhase(teeDataHeader & 0x00000FFF);
                }

                wStatus->showMessage(QString(tr("Temp: %1, Phase: %2, Frame %3 ").arg(m_tnTEETemperature).arg(m_tnTEEPhase).arg(frameID)));
            }
        }
    }
}

double PortaDemo::RawMiniTeeToTemperature(int voltage)
{
    //const double a = 0.0029365;
    //const double b = -13.113;
    //const double c = 14665.2;
    return voltage;//a * pow((double)voltage, 2) + b * voltage + c;
}

double PortaDemo::RawMiniTnTeeToPhase(int voltage)
{
    return voltage * 0.06667 - 43.33;
}
