#include "stdafx.h"
#include "PortaDisplay.h"

// interrupt for when a new processed frame is ready
int newFrameInterrupt(void* param, int /*id*/, int header)
{
    PortaDisplay* pd = (PortaDisplay*)param;

    return pd->setImgData(header) ? 1 : 0;
}

PortaDisplay::PortaDisplay(QWidget* parent) : QGraphicsView(parent)
{
    QGraphicsScene* sc = new QGraphicsScene(this);
    setScene(sc);

    m_image = 0;
    m_index = -1;
    m_mode = -1;
}

PortaDisplay::~PortaDisplay()
{
}

// initializes the display
bool PortaDisplay::init(int index, int mode)
{
    int w, h;
    m_index = index;
    m_mode = mode;

    portaSetDisplayCallback(m_index, newFrameInterrupt, (void*)this);

    getOptDims(w, h);
    portaSetDisplayDimensions(m_index, w, h);
    setupBuffer(w, h);
    setSceneRect(0, 0, w, h);

    return true;
}

// resizes the display scene
void PortaDisplay::resizeEvent(QResizeEvent* event)
{
    if (portaIsConnected())
    {
        int w, h;
        getOptDims(w, h);
        portaSetDisplayDimensions(m_index, w, h);
        setupBuffer(w, h);
        setSceneRect(0, 0, w, h);
    }
    else
    {
        setSceneRect(0, 0, width(), height());
    }

    QGraphicsView::resizeEvent(event);
}

// draws the ultrasound image
void PortaDisplay::drawBackground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawBackground(painter, r);

    if (m_image)
    {
        painter->drawImage(0, 0, *m_image, 0, 0);
    }
    else
    {
        painter->fillRect(0, 0, width(), height(), Qt::black);
    }
}

// initializes the image buffer
void PortaDisplay::setupBuffer(int w, int h)
{
    int sz;

    sz = w * h * 4;

    if (m_image)
    {
        delete m_image;
    }

    m_image = new QImage(w, h, QImage::Format_RGB32);
}

// returns the optimal dimensions for scan conversion
void PortaDisplay::getOptDims(int& w, int& h)
{
    w = width();
    while (w % 4)
    {
        w--;
    }

    h = height();
    while (h % 4)
    {
        h--;
    }
}

double PortaDisplay::getSampleSize(int samplingFreq, int vsound)
{
    if (!samplingFreq)
    {
        return 0;
    }

    return (double)(1000000.0 * (double)vsound) / (2.0 * (double)samplingFreq);
}

int PortaDisplay::micronsToUSSamples(int microns, int samplingFreq, int vsound)
{
    double sampleSz = getSampleSize(samplingFreq, vsound);
    return (int)(microns / sampleSz);
}

int PortaDisplay::usSamplesToMicrons(int samples, int samplingFreq, int vsound)
{
    double sampleSz = getSampleSize(samplingFreq, vsound);
    return (int)(sampleSz * samples);
}
