#include "stdafx.h"
#include "TexoDemo.h"

#define MSG_TIMEOUT   3000          // ms
#define CINE_SIZE     128           // MB
#define MAXDAQFOLDERS 30            // number of DAQ downloads
#define DAQFOLDER     "D:/DAQRT/"

#define bound(y, h) (y < 0 ? ((-y > h) ? -h : y) : ((y > h) ? h : y))

TexoDemo::TexoDemo(QWidget* parent) : QMainWindow(parent), m_workerProcessor(this)
{
    setupUi(this);

    m_texoFwPath = FIRMWARE_PATH;
    m_daqFwPath = DAQ_FIRMWARE_PATH;
    m_cineSize = CINE_SIZE;
    m_scanlines = 64;
    m_channels = 32;
    m_display = m_scanlines;
    m_samplesPerLine = 0;
    m_BsamplesPerLine = 0;
    m_rf = 0;   // used in the Acq thread
    m_rfII = 0; // used in the worker thread
    m_b = 0;

    m_texoIsRunning = false;    // used in the worker thread
    m_runAmplio = true;         // turns the Texo image former on or off
    m_runBeamforming = false;    // turns the DAQ beamformer on or off
    m_amp = 100;                // scale rf data for display
    m_updateDAQdisplay = true;

    // daq params
    m_daqFrCount = 0;
    strcpy(m_daqpath, DAQFOLDER);
    CreateDirectoryA(m_daqpath, 0);
    m_daqchannels = 128;
    m_daqsamplesPerLine = 2080;
    m_daq80MHzSampling = false;
    m_daqdata = 0;
    m_bfdaq = 0;
    m_daqDownloadingMode = STORE_TO_DISK_ONLY;

    if (m_bfdaq)
    {
        setupControls();
    }
}

TexoDemo::~TexoDemo()
{
    delete[] m_daqdata;
    delete[] m_bfdaq;

    if (m_rf)
    {
        delete[] m_rf;
    }

    if (m_rfII)
    {
        delete[] m_rfII;
    }

    if (m_b)
    {
        delete[] m_b;
    }
    texoShutdown();
    daqReleaseMem();
}

// create action groups so menus can be checkable
void TexoDemo::setupControls()
{
    QActionGroup* agMain = new QActionGroup(this);
    agMain->addAction(mMain2);
    agMain->addAction(mMain3);
    agMain->addAction(mMain4);

    QActionGroup* agPCI = new QActionGroup(this);
    agPCI->addAction(mPCI2);
    agPCI->addAction(mPCI3);
    agPCI->addAction(mPCI4);
}

// allow user to change data path
void TexoDemo::getTexoFirmwarePath()
{
    bool ok = false;
    QString text = QInputDialog::getText(this, tr("Enter Texo Firmware Path"), tr("Path:"), QLineEdit::Normal, m_texoFwPath, &ok);

    if (ok && !text.isEmpty())
    {
        m_texoFwPath = text;
    }
}

// allow user to change data path
void TexoDemo::getDaqFirmwarePath()
{
    bool ok = false;
    QString text = QInputDialog::getText(this, tr("Enter DAQ Firmware Path"), tr("Path:"), QLineEdit::Normal, m_daqFwPath, &ok);

    if (ok && !text.isEmpty())
    {
        m_daqFwPath = text;
    }
}

// allow user to change cine size
void TexoDemo::setCineSize()
{
    bool ok = false;
    int cine = QInputDialog::getInt(this, tr("Enter Cine Size"), tr("Cine Size (MB):"), m_cineSize, 32, 512, 1, &ok);

    if (ok)
    {
        m_cineSize = cine;
    }
}

// initialize the ultrasound engine, populate probes, load lists
void TexoDemo::initHardware()
{
    int i;
    int usm = mMain2->isChecked() ? 2 : (mMain3->isChecked() ? 3 : 4);
    int pci = mPCI2->isChecked() ? 2 : (mPCI3->isChecked() ? 3 : 4);
    m_channels = (usm == 2) ? 32 : 64;
    bool texoReady = false, daqReady = false;

    // init Texo
    wStatus->showMessage(tr("Initializing ultrasound ..."), MSG_TIMEOUT);
    if (!texoInit(m_texoFwPath.toAscii(), pci, usm, 0, m_channels, 0, m_cineSize))
    {
        wStatus->showMessage(tr("Could not intitialize ultrasound"), MSG_TIMEOUT);
        return;
    }
    else
    {
        wStatus->showMessage(tr("Successfully intitialized ultrasound"), MSG_TIMEOUT);
        readProbes();

        texoSetCallback(onData, this);
    }

    // initially select a probe - assume DAQ plugged in on 3rd port - which is i=1
    for (i = 0; i < 3; i++)
    {
        if (i == 1)
        {
            continue;
        }

        if (texoGetProbeCode(i) != 31)
        {
            if (texoSelectProbe(texoGetProbeCode(i)))
            {
                texoActivateProbeConnector(i);
            }
            texoReady = true;
            break;
        }
    }

    if (!texoReady)
    {
        wStatus->showMessage(tr("No probe found on the first 2 ports."), MSG_TIMEOUT);
        return;
    }

    // open the DAQ connector
    texoForceConnector(3);

    // initialize DAQ
    wStatus->showMessage(tr("Initializing DAQ ... This may take few minutes ..."), MSG_TIMEOUT);
    daqConnect(0);
    if (!daqIsConnected())
    {
        QMessageBox mb;
        mb.setText(tr("DAQ is not connected or is not powered"));
        mb.exec();
    }
    else if (!daqIsInitialized())
    {
        daqSetFirmwarePath(m_daqFwPath.toAscii());
        wStatus->showMessage(tr("Programming DAQ ..."), MSG_TIMEOUT);

        if (!daqInit(m_daq80MHzSampling))
        {
            QMessageBox mb;
            mb.setText(tr("Initializaing DAQ Failed"));
            mb.exec();
        }
        else
        {
            wStatus->showMessage(tr("Successfully intitialized DAQ"), MSG_TIMEOUT);
            daqReady = true;
        }
    }
    else
    {
        daqReady = true;
    }

    // load parameters
    reloadParams();

    // Need to program the texo sequence before the daq sequence to get the external clk going
    if (texoReady)
    {
        programSequence();
    }

    // load DAQ sequence
    if (daqReady)
    {
        wStatus->showMessage(sequenceDAQ() ? tr("Successfully loaded the DAQ") : tr("Loading DAQ Sequence Failed"), MSG_TIMEOUT);
    }
   
    // select the data transfer method
    daqTransferMode(m_daqDownloadingMode);
}

// callback for new frames
int TexoDemo::onData(void* prm, unsigned char* data, int)
{
    // copy to local rf frame
    ((TexoDemo*)prm)->m_txoMutex.lock();
    memcpy(((TexoDemo*)prm)->m_rf, data + 4, ((TexoDemo*)prm)->m_scanlines * ((TexoDemo*)prm)->m_samplesPerLine * sizeof(short));
    ((TexoDemo*)prm)->m_txoMutex.unlock();

    return 1;
}

// start imaging
void TexoDemo::runImage()
{
    if (texoIsInitialized() && texoRunImage())
    {
        m_texoIsRunning = true;
        m_workerProcessor.start();
        wStatus->showMessage(tr(QString("Imaging Running @ %1Hz").arg(texoGetFrameRate()).toAscii()));
    }
}

// stop imaging
void TexoDemo::stopImage()
{
    if (texoIsInitialized() && texoStopImage())
    {
        m_texoIsRunning = false;
        m_workerProcessor.wait();
        wStatus->showMessage(tr("Imaging Stopped"));
        wCine->setRange(0, texoGetCollectedFrameCount() - 1);
        wCine->setValue(wCine->maximum());
    }
}

// read probes and setup controls
void TexoDemo::readProbes()
{
    int i;
    char name[80];
    QAbstractButton* wProbe = 0;

    for (i = 0; i < 3; i++)
    {
        if (i == 1)
        {
            continue;
        }

        wProbe = (i == 0) ? wProbe1 : wProbe2;

        if (texoGetProbeName(i, name, 80))
        {
            wProbe->setText(name);
            wProbe->setEnabled(true);
        }
        else
        {
            wProbe->setText("");
            wProbe->setEnabled(false);
        }
    }
}

// select first probe
void TexoDemo::selectProbe1()
{
    if (texoSelectProbe(texoGetProbeCode(0)))
    {
        if (texoActivateProbeConnector(0))
        {
            wStatus->showMessage(tr("Selected first probe connector"), MSG_TIMEOUT);
        }
    }

    texoForceConnector(3);
}

// select second probe
void TexoDemo::selectProbe2()
{
    if (texoSelectProbe(texoGetProbeCode(2)))
    {
        if (texoActivateProbeConnector(2))
        {
            wStatus->showMessage(tr("Selected second probe connector"), MSG_TIMEOUT);
        }
    }
    texoForceConnector(3);
}

void TexoDemo::cineScroll(int index)
{
    if (m_texoIsRunning)
    {
        fprintf(stderr, "cine scroll is only available when imaging is stopped.\n");
        return;
    }

    unsigned char* cine = texoGetCineStart(0);
    cine += (texoGetFrameSize() * index);
    memcpy(m_rf, cine + 4, m_scanlines * m_samplesPerLine * sizeof(short));
    processTexo();
    displayTexo();
}

void TexoDemo::onAmp(int value)
{
    chDisplay->setAmp(value);

    m_amp = value;

    processTexo();
    displayTexo();
}

void TexoDemo::onChannelDisplay(int)
{
}

void TexoDemo::onApplyParams()
{
    bool running = texoIsImaging() != 0;

    if (running)
    {
        texoStopImage();
        m_texoIsRunning = false;
        m_workerProcessor.wait();
    }

    // relaod params
    reloadParams();

    // reload the sequencer
    if (!programSequence())
    {
        wStatus->showMessage(tr("Error Programming Sequence"));
    }

    if (running)
    {
        texoRunImage();
        m_texoIsRunning = true;
        m_workerProcessor.start();
        wStatus->showMessage(tr(QString("Imaging Running @ %1Hz").arg(texoGetFrameRate()).toAscii()));
    }
}

bool TexoDemo::programSequence()
{
    int i;
    _texoTransmitParams tx;
    _texoReceiveParams rx;
    _texoLineInfo li;

    if (!texoIsInitialized())
    {
        return false;
    }

    // setup a fixed tgc
    texoClearTGCs();
    texoAddTGCFixed((double)m_txoParams.gain / 100.0);
    /*// setup a tgc curve
       _texoCurve *tgccrv = new _texoCurve();
       tgccrv->top=50;
       tgccrv->mid=75;
       tgccrv->btm=100;
       tgccrv->vmid=90;
       texoAddTGC(tgccrv, m_txoParams.depth * 1000, (double)m_txoParams.gain / 100.0);*/

    // setup the power
    texoSetPower(m_txoParams.power, 15, 15);
    // output both sync and clock for the DAQ
    texoSetSyncSignals(0, 1, 3);
    // setup callback
    daqSetCallback(daqCallback, this);

    //setup VCA
    _vcaInfo vcaInfo;
    vcaInfo.amplification = 32;
    vcaInfo.activetermination = 4;
    vcaInfo.inclamp = 1600;
    vcaInfo.LPF = 1;
    vcaInfo.lnaIntegratorEnable = 1;
    vcaInfo.pgaIntegratorEnable = 1;
    vcaInfo.hpfDigitalEnable = 0;
    vcaInfo.hpfDigitalValue = 10;
    texoSetVCAInfo(vcaInfo);

    // start a new sequence
    if (!texoBeginSequence())
    {
        return false;
    }

    // set transmit parameters
    tx.angle = m_txoParams.angle * 1000;
    tx.aperture = m_txoParams.txaperture;
    tx.focusDistance = m_txoParams.focus * 1000;
    tx.frequency = m_txoParams.freq * 1000 * 1000;
    tx.speedOfSound = m_txoParams.vsound;
    tx.useManualDelays = 0;
    tx.useMask = 0;
    tx.tableIndex = -1;
    tx.sync = 0;
    strcpy_s(tx.pulseShape, m_txoParams.pulse.toLocal8Bit());
    tx.txRepeat = 0;
    tx.txDelay = 100;

    // set receive parameters
    rx.angle = m_txoParams.angle * 1000;
    rx.aperture = (m_txoParams.rxaperture < m_channels) ? m_txoParams.rxaperture : m_channels;
    rx.maxApertureDepth = 30000;
    rx.acquisitionDepth = m_txoParams.depth * 1000;
    rx.saveDelay = m_txoParams.delay * 1000;
    rx.speedOfSound = m_txoParams.vsound;
    rx.channelMask[0] = rx.channelMask[1] = 0xFFFFFFFF;
    rx.applyFocus = m_txoParams.bfdelay;
    rx.useManualDelays = 0;
    rx.decimation = m_txoParams.decim;
    rx.customLineDuration = 0;
    rx.lgcValue = 0;
    rx.tgcSel = 0;
    rx.tableIndex = -1;
    rx.numChannels = m_channels;
    rx.weightType = 1;

    // line density settings
    m_scanlines = m_txoParams.numLines;

    int numelements = 128;
    double virtualElmStep = double(numelements) / double(m_scanlines);
    // create the sequence for standard imaging
    for (i = 0; i < m_scanlines; i++)
    {
        tx.centerElement = i * virtualElmStep + 0.5;
        rx.centerElement = i * virtualElmStep + 0.5;

        if (!texoAddLine(tx, rx, li))
        {
            return false;
        }

        m_samplesPerLine = li.lineSize / 2;
    }

    // plane wave(s) following standard imaging
    if (m_txoParams.planeWave > 0)
    {
        // set plane wave transmit without focusing with full aperture
        tx.aperture = 128;
        tx.useManualDelays = 1;
        tx.angle = 0;
        memset(tx.manualDelays, 0, sizeof(int) * 128);
        // center the aperture
        tx.centerElement = (numelements / 2) + 0.5;
        // turn on the sync for the plane wave to trigger DAQ collection
        tx.sync = 1;
        // do not acquire texo data during the plane wave
        rx.acquisitionDepth = 0;
        // adjust the line duration if triggering DAQ in plane wave mode
        rx.customLineDuration = 100000; // 100 usec

        for (i = 0; i < m_txoParams.planeWave; i++)
        {
            if (!texoAddLine(tx, rx, li))
            {
                return false;
            }
        }
    }

    // finish the sequence
    if (texoEndSequence() == -1)
    {
        return false;
    }

    initAmplio();

    m_dispMutex.lock();
    bDisplay->init(m_scanlines, m_BsamplesPerLine);
    m_dispMutex.unlock();

    return true;
}

bool TexoDemo::initAmplio()
{
    // initialize IQ demodulation
    if (!amplioInitRfToB(m_amplioParams.startFreq, m_amplioParams.stopFreq, m_amplioParams.cutoffFreq,
            m_amplioParams.samplingFreq, m_samplesPerLine, m_amplioParams.useCompression, m_amplioParams.dynamicRg, m_amplioParams.reject)) // load the demodulation and compression tables
    {
        fprintf(stderr, "could not initialize amplio\n");
        return false;
    }

    // rf to bpr conversion parameters
    int decimFactor = 1 << m_amplioParams.decimation;
    m_BsamplesPerLine = m_samplesPerLine / decimFactor;

    // alocate memories
    allocateBuffers(m_scanlines * m_samplesPerLine, m_scanlines * m_BsamplesPerLine);

    return true;
}

void TexoDemo::processTexo()
{
    if (!m_rf || !m_b || !m_rfII)
    {
        return;
    }

    m_txoMutex.lock();
    memcpy(m_rfII, m_rf, m_scanlines * m_samplesPerLine * sizeof(short));
    m_txoMutex.unlock();

    int decimation = m_amplioParams.decimation; // 0: sampling freq, 1:sampling freq/2, 2:sampling freq/4, ...
    if (m_runAmplio)    // generate B image
    {
        // process each line separately
        for (int i = 0; i < m_scanlines; i++)
        {
            // read one RF line
            short* rfline = &m_rfII[i * m_samplesPerLine];
            unsigned char* bline = &m_b[i * m_BsamplesPerLine];
            amplioProcessRfToB(rfline, decimation, bline);
        }
    }
    else                // generate RF image
    {
        int skip = 1 << decimation;
        for (int i = 0; i < m_scanlines; i++)
        {
            short* rfline = &m_rfII[i * m_samplesPerLine];
            unsigned char* bline = &m_b[i * m_BsamplesPerLine];
            for (int j = 0; j < m_BsamplesPerLine; j++)
            {
                bline[j] = static_cast< unsigned char >(127 + bound(rfline[j * skip] * m_amp / 100 / 6, 127));
            }
        }
    }
}

bool TexoDemo::sequenceDAQ()
{
    daqRaylinePrms rlprms;
    daqSequencePrms seqprms;

    rlprms.channels = m_daqParams.channels;
    rlprms.gainDelay = m_daqParams.gainDelay;
    rlprms.gainOffset = m_daqParams.gainOffset;
    rlprms.rxDelay = m_daqParams.rxDelay;

    // sampling and decimation
    if (m_daq80MHzSampling)
    {
        rlprms.sampling = 80;   // DAQ sampling frequency 80 -> 80 [MHz]
        rlprms.decimation = 0;  // no decimation for 80MHz sampling
    }
    else
    {
        rlprms.sampling = 40;   // DAQ sampling frequency 40 -> 40 [MHz]
        rlprms.decimation = static_cast< unsigned char >(m_daqParams.decimation);  // Fs = sampling / (1+decimation) e.g. decimation = 1 -> Fs=20 MHz
    }
    rlprms.lineDuration = 100;  // line duration in micro seconds
    rlprms.numSamples = m_daqsamplesPerLine; // at 40MHz

    seqprms.freeRun = m_daqParams.freeRun;
    seqprms.hpfBypass = m_daqParams.hpfBypass;
    seqprms.divisor = static_cast< unsigned char >(m_daqParams.divisor);                  // data size = 16GB / 2^divisor
    seqprms.externalTrigger = m_daqParams.externalTrigger;  // sync with Texo transmits
    seqprms.externalClock = m_daqParams.externalClock;      // set to true if external clock is provided
    seqprms.fixedTGC = m_daqParams.fixedTGC;
    seqprms.lnaGain = m_daqParams.lnaGain;                  // 0:16dB, 1:18dB, 2:21dB
    seqprms.pgaGain = m_daqParams.pgaGain;                  // 0:21dB, 1:24dB, 2:27dB, 3:30dB
    seqprms.biasCurrent = m_daqParams.biasCurrent;          // 0,1,2,...,7

    if (seqprms.fixedTGC)
    {
        seqprms.fixedTGCLevel = 100;
    }
    else
    {
        // set TGC curve
        daqTgcSetX(2, 0.75f);
        daqTgcSetX(1, 0.5f);
        daqTgcSetX(0, 0.25f);

        daqTgcSetY(2, 0.75f);
        daqTgcSetY(1, 0.5f);
        daqTgcSetY(0, 0.25f);
    }

    if (!daqRun(seqprms, rlprms))
    {
        return false;
    }

    return true;
}

void TexoDemo::daqCallback(void* prm, int, ECallbackSources src)
{
    if (!((TexoDemo*)prm)->m_texoIsRunning)
    {
        return;
    }

    char path[256];
    strcpy(path, ((TexoDemo*)prm)->m_daqpath);
    char subdir[128];

    if (src == eCbBuffersFull)
    {
        // allocate the buffer
        ((TexoDemo*)prm)->allocateDAQBuffers(((TexoDemo*)prm)->m_daqchannels, ((TexoDemo*)prm)->m_daqsamplesPerLine);

        // init the display (technically this only needs to be done if the numsamples is changed)
        if (((TexoDemo*)prm)->m_updateDAQdisplay)
        {
            ((TexoDemo*)prm)->m_dispMutex.lock();
            ((TexoDemo*)prm)->chDisplay->init(((TexoDemo*)prm)->m_daqchannels, ((TexoDemo*)prm)->m_daqsamplesPerLine);
            ((TexoDemo*)prm)->m_dispMutex.unlock();
        }

        // create a folder for the new DAQ data
        sprintf(subdir, "%d/", ((TexoDemo*)prm)->m_daqFrCount);
        strcat(path, subdir);
        CreateDirectoryA(path, 0);
        ((TexoDemo*)prm)->incDaqCounter();
        // download the DAQ data into a folder
        daqDownload(path);
        // load the first DAQ frame from the folder to memory
        int numFrames = ((TexoDemo*)prm)->loadDaqData(path, 0);
        // send the first DAQ frame for display
        ((TexoDemo*)prm)->displayDAQ();
        // display the rest of DAQ frames
        for (int i = 1; i < numFrames; i++)
        {
            // load i'th DAQ frame
            ((TexoDemo*)prm)->loadDaqData(path, i);
            // display i'th DAQ frame
            ((TexoDemo*)prm)->displayDAQ();
        }

        // update the number of samples in case depth has changed
        if (((TexoDemo*)prm)->m_daqsamplesPerLine == ((TexoDemo*)prm)->m_samplesPerLine)
        {
            ((TexoDemo*)prm)->m_updateDAQdisplay = false;
        }
        else
        {
            ((TexoDemo*)prm)->m_daqsamplesPerLine = ((TexoDemo*)prm)->m_samplesPerLine;
            ((TexoDemo*)prm)->m_updateDAQdisplay = true;
        }

        // Reload the DAQ sequence now,
        // This way we make sure call back is not called during the processing
        ((TexoDemo*)prm)->sequenceDAQ();
    }
}

// allow data to be loaded from file with a frame offset
int TexoDemo::loadDaqData(char* path, int frame)
{
    // mapping the daq channel index to transducer element index
    char channelIndex[128] =
    {
        0, 16, 32, 48, 64, 80, 96, 112, 1, 17, 33, 49, 65, 81, 97, 113, 2, 18, 34, 50, 66, 82, 98, 114,
        3, 19, 35, 51, 67, 83, 99, 115, 4, 20, 36, 52, 68, 84, 100, 116, 5, 21, 37, 53, 69, 85, 101, 117,
        6, 22, 38, 54, 70, 86, 102, 118, 7, 23, 39, 55, 71, 87, 103, 119, 8, 24, 40, 56, 72, 88, 104, 120,
        9, 25, 41, 57, 73, 89, 105, 121, 10, 26, 42, 58, 74, 90, 106, 122, 11, 27, 43, 59, 75, 91, 107,
        123, 12, 28, 44, 60, 76, 92, 108, 124, 13, 29, 45, 61, 77, 93, 109, 125, 14, 30, 46, 62, 78, 94,
        110, 126, 15, 31, 47, 63, 79, 95, 111, 127
    };
    
    int frames = 0;
    int channel = 0;
    int samples = 0;

    if (m_daqDownloadingMode == STORE_TO_DISK_ONLY)
    {
        FILE* fp;
        QString file, msg;

        QDir dir(path);
        QStringList files = dir.entryList(QStringList("*.daq"));

        for (int i = 0; i < files.count(); i++)
        {
            file = dir.path() + "/" + files[i];
            fp = fopen(file.toLocal8Bit(), "rb");
            if (fp)
            {
                fread(&channel, sizeof(int), 1, fp);
                fread(&frames, sizeof(int), 1, fp);
                fread(&samples, sizeof(int), 1, fp);

                // check for error
                if (frames <= 0 || samples <= 0 || samples != m_daqsamplesPerLine || channel >= m_daqchannels)
                {
                    fclose(fp);
                    wStatus->showMessage(tr("Error loading DAQ data"));
                    return 0;
                }

                // load channel data
                fseek(fp, (samples * frame * sizeof(short)), SEEK_CUR);
                fread(m_daqdata + (channelIndex[channel] * samples * sizeof(short)), sizeof(short), samples, fp);

                fclose(fp);
            }
        }
    }
    else
    {
        for (int i = 0; i < 128; i++)
        {
            // read header
            int* header = reinterpret_cast< int* >(daqGetDataPtr(i));
            try
            {
                channel = header[0];
                frames = header[1];
                samples = header[2];

                // read data
                short int* data = reinterpret_cast< short int* >(daqGetDataPtr(i));
                memcpy(reinterpret_cast< void* >(m_daqdata + (channelIndex[channel] * samples * sizeof(short int))),
                    reinterpret_cast< void* >(data + DAQ_FILE_HEADER_SIZE_BYTES / sizeof(short int) + samples *
                                              frame), samples * sizeof(short int));
            }
            catch (...)
            {
            }
        }
    }

    return frames;
}

// display DAQ frame
void TexoDemo::displayDAQ()
{
    m_dispMutex.lock();

    if (m_runBeamforming)
    {
        beamformDAQ();
        chDisplay->setImgData(m_bfdaq);
    }
    else
    {
        chDisplay->setImgData(m_daqdata);
    }

    m_dispMutex.unlock();
}

// display Texo frame
void TexoDemo::displayTexo()
{
    m_dispMutex.lock();
    bDisplay->setImgData((char*)m_b);
    m_dispMutex.unlock();
}

void TexoDemo::incDaqCounter()
{
    // only keep a limited number of daq folders
    m_daqFrCount = (m_daqFrCount + 1) % MAXDAQFOLDERS;
}

void TexoDemo::onSave()
{
    if (m_texoIsRunning)
    {
        fprintf(stderr, "saving is only available when imaging is stopped.\n");
        return;
    }

    int numFrames = texoGetCollectedFrameCount();
    int frameSize = texoGetFrameSize();

    if (numFrames < 1)
    {
        fprintf(stderr, "no frame available\n");
        return;
    }

    if (frameSize < 1)
    {
        fprintf(stderr, "frame size is zero\n");
        return;
    }

    FILE* fp = fopen("data.txo", "wb+");
    if (!fp)
    {
        fprintf(stderr, "could not open the file\n");
        return;
    }

    fprintf(stdout, "Storing data in data.txo:\n\n");
    fprintf(stdout, "-number of RF scan lines = %d \n", m_scanlines);
    fprintf(stdout, "-number of RF samples = %d \n", m_samplesPerLine);
    fprintf(stdout, "-frame size = %d bytes\n", frameSize);
    fprintf(stdout, "-number of frames = %d \n\n", numFrames);

    fwrite(texoGetCineStart(0), frameSize, numFrames, fp);
    fclose(fp);

    wStatus->showMessage(tr("Successfully stored data"));

    return;
}

void TexoDemo::allocateBuffers(int szRF, int szB)
{
    if (m_rf)
    {
        delete[] m_rf;
    }

    if (m_rfII)
    {
        delete[] m_rfII;
    }

    if (m_b)
    {
        delete[] m_b;
    }

    m_rf = new short[szRF];
    m_rfII = new short[szRF];
    m_b = new unsigned char[szB];

    resetBuffers(szRF, szB);
}

void TexoDemo::allocateDAQBuffers(int channels, int samples)
{
    if (m_daqdata)
    {
        delete[] m_daqdata;
    }

    if (m_bfdaq)
    {
        delete[] m_bfdaq;
    }

    m_daqdata = new char[channels * samples * sizeof(short)];
    m_bfdaq = new char[channels * samples * sizeof(short)];
}

void TexoDemo::resetBuffers(int szRF, int szB)
{
    if (m_rf)
    {
        memset(m_rf, 0, szRF * sizeof(short));
    }

    if (m_rfII)
    {
        memset(m_rfII, 0, szRF * sizeof(short));
    }

    if (m_b)
    {
        memset(m_b, 0, szB);
    }
}

void TexoDemo::reloadParams()
{
    /////////////// load Texo params ///////////////

    m_txoParams.numLines = wParamsTXO->item(0, 0)->text().toInt();
    m_txoParams.angle = wParamsTXO->item(1, 0)->text().toInt();
    m_txoParams.vsound = wParamsTXO->item(2, 0)->text().toInt();
    m_txoParams.freq = wParamsTXO->item(3, 0)->text().toInt();
    m_txoParams.txaperture = wParamsTXO->item(4, 0)->text().toInt();
    m_txoParams.power = wParamsTXO->item(5, 0)->text().toInt();
    m_txoParams.pulse = wParamsTXO->item(6, 0)->text();
    m_txoParams.focus = wParamsTXO->item(7, 0)->text().toInt();
    m_txoParams.rxaperture = wParamsTXO->item(8, 0)->text().toInt();
    m_txoParams.gain = wParamsTXO->item(9, 0)->text().toInt();
    m_txoParams.depth = wParamsTXO->item(10, 0)->text().toInt();
    m_txoParams.delay = wParamsTXO->item(11, 0)->text().toInt();
    m_txoParams.decim = wParamsTXO->item(12, 0)->text().toInt();
    m_txoParams.bfdelay = (wParamsTXO->item(13, 0)->checkState() == Qt::Unchecked) ? false : true;
    m_txoParams.planeWave = wParamsTXO->item(14, 0)->text().toInt();

    /////////////// load amplio params ///////////////

    // run amplio or not
    m_runAmplio = (wParamsAMP->item(0, 0)->checkState() == Qt::Unchecked) ? false : true;
    // IQ demodulation parameters
    m_amplioParams.samplingFreq = 40000000 >> m_txoParams.decim;
    m_amplioParams.startFreq = wParamsAMP->item(1, 0)->text().toInt() * 1000000;
    m_amplioParams.stopFreq = wParamsAMP->item(2, 0)->text().toInt() * 1000000;
    m_amplioParams.cutoffFreq = wParamsAMP->item(3, 0)->text().toInt() * 1000000;
    // initialize Log Compression
    m_amplioParams.useCompression = 1;    // if set to zero no log compression will be applied
    m_amplioParams.reject = wParamsAMP->item(4, 0)->text().toInt(); // rejection in dB
    m_amplioParams.dynamicRg = wParamsAMP->item(5, 0)->text().toInt(); // dynamic range in dB
    m_amplioParams.decimation = 1; // for b iamage, 0: sampling freq, 1:sampling freq/2, 2:sampling freq/4, ...

    /////////////// load daq param ///////////////

    // turn on/off software beamforming
    m_runBeamforming = (wParamsDAQ->item(0, 0)->checkState() == Qt::Unchecked) ? false : true;

    // turn on all channels
    m_daqParams.channels[0] = 0xffffffff;
    m_daqParams.channels[1] = 0xffffffff;
    m_daqParams.channels[2] = 0xffffffff;
    m_daqParams.channels[3] = 0xffffffff;

    m_daqParams.gainDelay = 0;
    m_daqParams.gainOffset = 0;
    m_daqParams.rxDelay = wParamsDAQ->item(1, 0)->text().toInt();
    m_daqParams.decimation = wParamsDAQ->item(2, 0)->text().toInt();
    m_daqParams.freeRun = 0;
    m_daqParams.hpfBypass = 0;
    m_daqParams.divisor = wParamsDAQ->item(3, 0)->text().toInt();
    // data size = 16GB / 2^divisor
    m_daqParams.externalTrigger = 1;    // sync with Texo transmits
    m_daqParams.externalClock = 0;      // set to true if external clock is provided
    m_daqParams.fixedTGC = (wParamsDAQ->item(4, 0)->checkState() == Qt::Unchecked) ? 0 : 1;
    m_daqParams.lnaGain = wParamsDAQ->item(5, 0)->text().toInt();            // 0:16dB, 1:18dB, 2:21dB
    m_daqParams.pgaGain = wParamsDAQ->item(6, 0)->text().toInt();            // 0:21dB, 1:24dB, 2:27dB, 3:30dB
    m_daqParams.biasCurrent = wParamsDAQ->item(7, 0)->text().toInt();        // 0,1,2,...,7
}

void TexoDemo::Processor::run()
{
    while (m_txo->m_texoIsRunning)
    {
        // process the image
        m_txo->processTexo();
        // display the image
        m_txo->displayTexo();
    }
}

void TexoDemo::beamformDAQ()
{
    int const pitch = 300;    // microns
    int const c = 1540;       // speed of sound m/s
    int const Fs = 40;        // sampling freq MHz
    int const sampleSpacing = c / Fs / 2;     // microns
    int const Fnum = 2;
    int const maxAprSz = 128;
    short* in = (short*)m_daqdata;
    short* out = (short*)m_bfdaq;

    for (int i = 0; i < m_daqchannels; i++)
    {
        for (int j = 0; j < m_daqsamplesPerLine; j++)
        {
            // find sample location in microns
            int Z = j * sampleSpacing;
            int X = i * pitch;

            // calculate the apr based on the F number
            int a = static_cast< int >(Z / (2 * Fnum));
            int hlfAprSz = a / pitch;

            // check to make sure we do not go beyound max apr
            if ((2 * hlfAprSz + 1) > maxAprSz)
            {
                hlfAprSz = maxAprSz / 2 - 1;
            }

            // calc delays based on sample depth and receive aperture
            int startApr = (i - hlfAprSz);
            int endApr = (i + hlfAprSz);

            if (startApr < 0)
            {
                startApr = 0;
            }
            if (endApr > (m_daqchannels - 1))
            {
                endApr = m_daqchannels - 1;
            }

            // caculate delays and sum
            int sum = 0;
            for (int chInd = startApr; chInd <= endApr; chInd++)    // aperture indices
            {
                int X1 = chInd * pitch;
                int chDelay = static_cast< int >((Z + sqrt(double(Z) * Z + double(X - X1) * (X - X1))) / c * Fs);

                if (chDelay >= m_daqsamplesPerLine)
                {
                    continue;
                }

                sum += in[chInd * m_daqsamplesPerLine + chDelay];
            }

            out[i * m_daqsamplesPerLine + j] = static_cast< short >(sum);
        }
    }
}
