#include "stdafx.h"
#include "TexoViewImg.h"
#include "math.h"

#define bound(y, h) (((y) < 0) ? ((-(y) > h) ? -h : y) : (((y) > h) ? h : y))

TexoViewImg::TexoViewImg(QWidget* parent) : QGraphicsView(parent)
{
    QGraphicsScene* sc = new QGraphicsScene(this);
    setScene(sc);

    m_width = 0;
    m_height = 0;
    m_buffer = 0;
    m_bufferImg = 0;
    m_image = 0;
    m_amp = 100;
    m_transpose = false;
}

TexoViewImg::~TexoViewImg()
{
    if (m_buffer)
    {
        delete[] m_buffer;
    }

    if (m_bufferImg)
    {
        delete[] m_bufferImg;
    }
}

bool TexoViewImg::init(int X, int Y, bool transpose)
{
    if (X * Y == 0)
    {
        return false;
    }

    m_width = X;
    m_height = Y;
    m_transpose = transpose;

    if (m_buffer)
    {
        delete[] m_buffer;
    }

    if (m_bufferImg)
    {
        delete[] m_bufferImg;
    }

    if (m_image)
    {
        delete m_image;
    }

    // RGB buffer for the image
    int sz = m_width * m_height;
    m_buffer = new char[sz * sizeof(short)];
    m_bufferImg = new unsigned char[sz * 4];

    if (!m_transpose)
    {
        m_image = new QImage(m_bufferImg, m_height, m_width, QImage::Format_RGB32);
    }
    else
    {
        m_image = new QImage(m_bufferImg, m_width, m_height, QImage::Format_RGB32);
    }

    setSceneRect(0, 0, width(), height());

    return true;
}

void TexoViewImg::resizeEvent(QResizeEvent* e)
{
    setSceneRect(0, 0, e->size().width(), e->size().height());
    QGraphicsView::resizeEvent(e);
}

void TexoViewImg::drawBackground(QPainter* painter, const QRectF& r)
{
    QGraphicsView::drawBackground(painter, r);

    if (m_image)
    {
        if (!m_transpose)
        {
            painter->drawImage(QRect(0, 0, width(), height()), *m_image, QRect(0, 0, m_height, m_width), Qt::MonoOnly);
        }
        else
        {
            painter->drawImage(QRect(0, 0, width(), height()), *m_image, QRect(0, 0, m_width, m_height), Qt::MonoOnly);
        }
    }
    else
    {
        painter->fillRect(0, 0, width(), height(), Qt::black);
    }
}

void TexoViewImg::convertRF2Img(char* in, unsigned char* out)
{
    short* data = (short*)(in);
    int* img = (int*)(out);

    for (int j = 0; j < m_width; j++)
    {
        for (int i = 0; i < m_height; i++)
        {
            int rf = data[j * m_height + i];
            int val = 127 + bound(rf * m_amp / 100, 127);
            if (!m_transpose)
            {
                img[j * m_height + i] = (val + (val << 8) + (val << 16));
            }
            else
            {
                img[i * m_width + j] = (val + (val << 8) + (val << 16));
            }
        }
    }
}

bool TexoViewImg::setImgData(char* data)
{
    if (!m_buffer || !m_bufferImg || !m_image)
    {
        return false;
    }

    memcpy(m_buffer, data, m_width * m_height * sizeof(short));
    convertRF2Img(m_buffer, m_bufferImg);

    scene()->invalidate();
    return true;
}

void TexoViewImg::setAmp(int amp)
{
    m_amp = amp;
    convertRF2Img(m_buffer, m_bufferImg);
    scene()->invalidate();
}
