%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Author: Christopher Hunt <chunt11@jhu.edu>
%         Matthew Walmer <mwalmer3@jhu.edu>
% Date: 4 May 2016
% Description: Performs robot-robot calibration using SURF image features and
%              depth data
% Input: dash_hand_files  -> string of the path to Dash's hand-eye calibration data
%        jerry_hand_files -> string of the path to Jerry's hand-eye calibration data
%        sq_size          -> size of a square of the checkerboard ( in mm )
%        dash_surf_files  -> string of the path to Dash's SURF calibration data
%        jerry_surf_files -> string of the path to Jerry's SURF calibration data
%        min_shared       -> overlapping feature threshold to discard bad matches
% Output: b1Hb2 -> The transformation from robot 2 (Dash) to robot 1 (Jerry)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% dependencies

clear all; clc;
addpath( strcat( pwd, '/util' ) );

%% global variables, change these for a different calibration set

dash_hand_files = strcat( pwd, '/Depth and Feature data/Checkerboard Images/16-04-18/Dash' ); % path to Dash hand-eye calibration files
jerry_hand_files = strcat( pwd, '/Depth and Feature data/Checkerboard Images/16-04-18/Jerry' ); % path to Jerry hand-eye calibration files
sq_size = 30; % size of one side of a square of the calibration checkerboard (in mm)          

dash_surf_files = strcat( pwd, '/Depth and Feature data/16-04-18/Dash' ); % path to Dash SURF calibration files
jerry_surf_files = strcat( pwd, '/Depth and Feature data/16-04-18/Jerry' ); % path to Jerry SURF calibration files
min_shared = 50; % number of features required to overlap between each frame for validity

%% perform hand-eye calibrations

disp( 'Hand eye calibration for Jerry...' );
[ g1Hc1, g1Hc1_err, ~, ~, cp1 ] = calibrate_handeye( jerry_hand_files, sq_size );
disp( 'Hand eye calibration for Dash...' );
[ g2Hc2, g2Hc2_err, ~, ~, cp2 ] = calibrate_handeye( dash_hand_files, sq_size );

%% generating point clouds for both robots

[ jerry_cloud, jerry_points, jerry_features, b1Hg1 ] = generate_surf_cloud( jerry_surf_files, cp1 );
[ dash_cloud, dash_points, dash_features, b2Hg2 ] = generate_surf_cloud( dash_surf_files, cp2 );

%% correlating the point clouds

disp( 'Correlating point clouds...' );
num_poses_jerry = length( jerry_cloud );
num_poses_dash = length( dash_cloud );

% finding matching features
feature_pairs = cell( num_poses_jerry, num_poses_dash ); % feature_pairs{i,j} corresponds 
                                                         % to the pair that relate 
                                                         % the ith Jerry feature set to
                                                         % the jth Dash feature set
valid = logical( zeros( num_poses_jerry, num_poses_dash ) ); %#ok<LOGL> determines if this is a valid set 
for i = 1:num_poses_jerry
    for j = i:num_poses_dash
        feature_pairs{ i, j } = matchFeatures( jerry_features{ i }, dash_features{ j },...
                                               'Unique', true,...
                                               'Method', 'Exhaustive',...
                                               'MaxRatio', 0.6 ); % find correspondences
        valid( i, j ) = size( feature_pairs{ i, j }, 1 ) > min_shared;
    end
end
num_valid = sum( valid( : ) );

%% get camera to camera transformation

disp( 'Calculating the the camera to camera transformation...' );
c1Hc2 = zeros( 4, 4, num_valid );
which_img = zeros(num_valid,2);
count = 1;
for i = 1:num_poses_jerry
    for j = i:num_poses_dash
        if valid( i, j )
            try %wrapping in try-catch in case there aren't enough inliers
                fprintf( 'Calculating c1Hc2 for Jerry %d, Dash %d...\n', i, j );
                % rearrange so there is correspondence
                jc = jerry_cloud{ i }( feature_pairs{ i, j }( :, 1 ), : );
                jp = jerry_points{ i }( feature_pairs{ i, j }( :, 1 ), : );
                
                dc = dash_cloud{ j }( feature_pairs{ i, j }( :, 2 ), : );
                dp = dash_points{ j }( feature_pairs{ i, j }( :, 2 ), : );
                
                % uncomment to visualize the SURF points on the undistorted image
                % I1 = undistortImage( imread( sprintf('%s/color/cImg_%d.png', jerry_surf_files, i) ), cp1 );
                % I2 = undistortImage( imread( sprintf('%s/color/cImg_%d.png', dash_surf_files, j) ), cp2 );
                % 
                % h1 = figure();
                % imshow( I1 );
                % h2 = figure();
                % imshow( I2 );
                % for k = 1:length( jp )
                %     set( 0, 'CurrentFigure', h1 ); hold on;
                %     plot( jp(k, 1), jp(k, 2), 'ro', 'markers', 8 );
                %     hold off;
                %     set( 0, 'CurrentFigure', h2 ); hold on;
                %     plot( dp(k, 1), dp(k, 2), 'co', 'markers', 8 );
                %     hold off;
                % end             
                % prune outliers
                [ ~, inliers ] = estimateFundamentalMatrix( jp, dp,...
                                                            'Method', 'LMedS',...
                                                            'InlierPercentage', 40,...
                                                            'NumTrials', 2000,...
                                                            'DistanceThreshold', 1e-4,...
                                                            'Confidence', 99 );
                jc_in = jc( inliers, : );
                dc_in = dc( inliers, : );
                
                fprintf( '%d (of %d) inliers  used for registration...\n',...
                         sum( inliers ), length(inliers) );
                
                % get transformation
                [d, Z, t] = procrustes( dc_in, jc_in, 'reflection', false, 'scaling', false );
                if d > 0.1 % residual distance threshold
                    fprintf( 'Sum of square distances of %.04f is too high. Ignoring this pair...\n', d );
                    valid( i, j ) = 0;
                else
                    % uncomment to view plots of cloud, reprojected cloud using the computed transform                    
                    % figure();
                    % plot3( jc(:, 1), jc(:, 2), jc(:,3), 'ro' );
                    % hold on;
                    % plot3( Z(:, 1), Z(:, 2), Z(:, 3), 'bo' );
                    % hold off;
                    c1Hc2( :, :, count ) = [ t.b * [t.T, t.c(1,:)' ] ; 0 0 0 1 ]; % homogenize
                    which_img( count, : ) = [ i, j ];
                    count = count + 1;
                end
            catch err
                disp( 'Not enough inliers. Skipping this pair...' );
                valid( i, j ) = 0;
            end
        end
    end
end

% update validity and trim c1Hc2
num_valid = sum( valid( : ) );
c1Hc2 = c1Hc2( :, :, 1:num_valid );

%% calculate base to base transformation

disp('Calculating the base to base transformation...');
b1Hb2_m = zeros( 4, 4, num_valid );
for i = 1:num_valid
    b1Hb2_m( :, :, i ) = b1Hg1( :, :, which_img(i,1) ) * g1Hc1 *...
                         c1Hc2( :, :, i ) *...
                         inv( g2Hc2 )  *  inv( b2Hg2( :, :, which_img(i,2) ) );
end
b1Hb2_dev = std( b1Hb2, 0, 3 );
b1Hb2 = mean( b1Hb2, 3 );