%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Author: Christopher Hunt <chunt11@jhu.edu>
% Date: 4 May 2016
% Description: Generates the 3D point clouds from SURF features found in N
%              2D images
% Input: filepath   -> string of the path to the robot's calibration data
%        cam_params -> the camera parameter object for the images taken
% Output: surf_clouds   -> cell array of N 3D point clouds (Mx3) of SURF interest points
%                          extrapolated into 3D space for N robot poses
%         surf_pts      -> cell array of N matrices (Mx2) of SURF interest point
%                          locations in the 2D RGB image for N robot poses
%         surf_features -> cell of array of N Mx64 feature matrices,
%                          each row corresponding to an interest point in
%                          the 2D RGB image
%         r_poses       -> A 4x4xN matrix of homogenous robot poses
% 
% Note: In the above notation, M is the number of interest points located
%       in that particular image number (i.e. M is not constant from pose to pose)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ surf_clouds, surf_pts, surf_features, r_poses ] = generate_surf_cloud( filepath, cam_params )
    %% Loading calibration data
    
    % Loading and parsing calibration data
    disp( 'Loading and parsing calibration data...' );
    name_index = find( filepath == '/', 1, 'last' ) + 1;
    robot_name = filepath( name_index:end );
    robot_poses = xlsread( [ filepath sprintf( '/%sPoses.xlsx', robot_name ) ] );
    num_poses = size( robot_poses, 1 );
    
    % loading image data
    c_images = cell( 1, num_poses );
    d_images = cell( 1, num_poses );
    im_borders = zeros( num_poses, 4, 'uint16' ); % [ x_low, x_high, y_low, y_high ]
    r_poses = zeros( 4, 4, num_poses );
    for i = 1:num_poses
        cI = undistortImage( imread( [ filepath sprintf('/color/cImg_%d.png', i) ] ), cam_params );
        im_borders( i, : ) = mask_image( cI );
        c_images{i} = rgb2gray( cI );
        d_images{i} = undistortImage( imread( [ filepath sprintf('/depth/dImg_%d.png', i) ] ), cam_params );
        r_poses( :, :, i ) = [ eul2rotm( robot_poses( i, 4:6 ) .* ( pi / 180 ), 'ZYX' ),...
                               robot_poses( i, 1:3 )';...
                               0 0 0 1 ];
    end
    
    %% Generating 2D SURF points
    
    fprintf( 'Generating 2D SURF points for %s...\n', robot_name );
    surf_points = cell( 1, num_poses ); % holds each valid SURF point
    surf_features = cell( 1, num_poses ); % holds feature vector for each SURF point
    for i = 1:num_poses
        surf_points{i} = detectSURFFeatures( c_images{i},...
                                             'MetricThreshold', 200,...
                                             'NumOctaves', 3 );
        surf_features{i} = extractFeatures( c_images{i}, surf_points{i} );
    end
    
    %% Generating set of 3D SURF cloud
    
    fprintf( 'Generating 3D SURF cloud for %s...\n', robot_name );
    surf_clouds = cell( 1, num_poses );
    surf_pts = cell( 1, num_poses );
    K = cam_params.IntrinsicMatrix;
    for i = 1:num_poses
        
        % get rid of border points and features (noisy)
        in_left = surf_points{ i }.Location( :, 1 ) > im_borders( i, 1 ); % x (cols)
        in_right = surf_points{ i }.Location( :, 1 ) < im_borders( i, 2 );
        in_bottom = surf_points{ i }.Location( :, 2 ) > im_borders( i, 3 ); % y (rows)
        in_top = surf_points{ i }.Location( :, 2 ) < im_borders( i, 4 );
        
        idx = find( ( in_left + in_right + in_bottom + in_top ) == 4 );
        surf_features{ i } = surf_features{ i }( idx, : );
        surf_pts{ i } = surf_points{ i }.Location( idx, : );
        
        p_x = double( round( surf_pts{ i }( :, 1 ) ) );
        p_y = double( round( surf_pts{ i }( :, 2 ) ) );
        
        z_ind = sub2ind( size( d_images{ i } ), p_y, p_x );
        z = double( d_images{ i }( z_ind ) );
        
        x = z .* ( p_x - K(3) ) ./ K(1);
        y = z .* ( p_y - K(6) ) ./ K(5);
        
        surf_clouds{ i } = [ x, y, z ];
        
        
        ind = all( surf_clouds{ i } == 0, 2 ); % indices of all zero values (if they exist)
        surf_clouds{ i }( ind, : ) = []; % remove them from the cloud
        surf_pts{ i }( ind, : ) = []; % remove them from the 2D
        surf_features{ i }( ind, : ) = []; % remove them from the feature set
    end
end