%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Author: Matthew Walmer <mwalmer3@jhu.edu>
% Date: 28 April 2016
% Description: Script to compute average marker pose
%
% Input Data File Formats:
%   robot_data_m_file contents:
%       bHg -> n x 1 cell, each containing a 4x4 homogenous transformation
%              matrix representing a robot gripper to base frame transf.
%       cHm -> n x 1 cell, each containing a 4x4 homogenous transformation
%              matrix representing a marker frame to camera frame transf.
%   robot_handeye:           
%       gHc -> 4 x 4 homogenous transformation matrix from the camera
%              frame to the gripper frame.
% Outputs:
%   pose  -> averaged marker pose with respect to the robot base frame as
%            a 4x4 homogenous transformation matrix
%   V     -> variance in the marker pose computed for each position
%   SD    -> standard deviation in the marker pose computed for each 
%            position
%   poses -> pose of the marker computed for each of the positions
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [pose, V, SD, poses] = ATK_get_marker_pose( robot_data_m_filename, handeye_filename )
% returns averaged marker pose computed in the robot base frame (bHm)
% based on the input data set and handeye transformation.
% Also returns variance and standard deviation of poses. The full set of 
% poses can be returned as well.

% LOAD INPUTS
load(robot_data_m_filename) %loads bHg and cHm
load(handeye_filename)      %loads gHc

% COMPUTE MARKER TO BASE
n = size(bHg,1);
poses = zeros(4,4,n);
for i = 1:n
    poses(:,:,i) = bHg{i} * gHc * cHm{i};
end

% MEAN AND VARIANCE
pose = mean(poses,3);
V = var(poses(1:3,:,:),0,3);
SD = V.^(1/2);

end

