%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Author: Christopher Hunt <chunt11@jhu.edu>
%         Matthew Walmer <mwalmer3@jhu.edu>
% Date: 31 March 2016
% Description: Performs hand-eye calibration of a single robot using a
%              calibration checkerboard
%
% Input:    filepath    -> string of the path to the calibration data
%                          location
%           square_size -> size of a square of the checkerboard ( in mm )
%
% Output:   gHe     -> The transformation from the camera space to the 
%                      gripper
%           err     -> The error associated with the calibration
%           bHw     -> Computed world frame (checkerboard) to robot base
%                      frame transformation
%           bHw_dev -> Set of world to base transformation for each robot 
%                      pose in the data set
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [ gHe, err, bHw, bHw_dev, cam_params ] = calibrate_handeye( filepath, square_size )
    %% Load and reformat end-effector pose data and generate file names
    disp( 'Loading and parsing calibration data...' );
    name_index = find( filepath == '/', 1, 'last' ) + 1;
    robot_name = filepath( name_index:end );
    robot_poses = xlsread( [ filepath sprintf( '/%sPoses.xlsx', robot_name ) ] );
    num_poses = size( robot_poses, 1 );
    
    calib_poses = zeros( 4, 4, num_poses );
    image_files = cell( 1, num_poses );
    for i = 1:num_poses
        %image_files{ i } = [ filepath sprintf( '/color/cImg_%d.bmp', i ) ]; % store image files
        image_files{ i } = [ filepath sprintf( '/color/cImg_%d.png', i ) ]; % store image files
        Rr = eul2rotm( robot_poses( i, 4:6 ) .* ( pi / 180 ), 'ZYX' ); % pose rotations
        tr = robot_poses( i, 1:3 )'; % pose translation
        calib_poses( :, :, i ) = [ Rr tr; 0 0 0 1 ]; % homogenous transformation
    end
    
    %%  Generate camera extrinsics
    disp( 'Generating and formatting camera extrinsics...' );
    cam_poses = zeros( 4, 4, num_poses );
    [ image_points, board_size, valid_images ] = detectCheckerboardPoints( image_files );
    world_points = generateCheckerboardPoints( board_size, square_size );
    cam_params = estimateCameraParameters( image_points, world_points, 'WorldUnits', 'mm' );
    count = 1;
    for i = 1:num_poses
        if ( valid_images( i ) )
            Rc = inv( cam_params.RotationMatrices( :, :, count ) ); % inverting because it works
            tc = cam_params.TranslationVectors( count, : )';
            cam_poses( :, :, i ) = [ Rc tc; 0 0 0 1 ];
            count = count + 1;
        end
    end
    
    %% Hand Eye Calibration
    disp( 'Generating the homogenous transformation...' );
    [ gHe, err ] = tsai_calibration( calib_poses( :, :, valid_images ),...
                                     cam_poses( :, :, valid_images ) );

    %% Compute Board WRT Robot Base (Average)
    num_images = sum( valid_images );
    bHwi = zeros( 4, 4, num_images );
    for i = 1:num_images
        if valid_images(i)
            eHw = cam_poses( :, :, i ); %world(board) to eye(camera)
            bHg = calib_poses( :, :, i ); %gripper to base
            % gHe = hand eye calibration - eye(camera) to gripper frame
            bHwi(:,:,i) = bHg * gHe * eHw;
        end
    end
    bHw_dev = std( bHwi, 0, 3 ); % get the standard deviation for board pose
    bHw = mean( bHwi, 3 ); % average board pose
 
end