%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Author: Matthew Walmer <mwalmer3@jhu.edu>
% Date: 1 April 2016
% Description: Script to evaluate error in different robot-robot
%              calibrations, using ground truth points created through
%              manual pointer allignment. Also plots error visualization.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
clear; clc;
addpath('util')

%SETTINGS
graph_it = true;
spin_it = false;
scaled_vectors = true;
vec_scale = 10;

%% Load Ground Truth Data
load('data/GT_pointer_data_16_04_24.mat')

%% Select Calibration

%Checkerboard Calibration (16-03-25 dataset) "SET A"
%H = [-0.999625865251467,-0.0232544236969541,-0.0146225553647442,1277.06910205243;0.0233232610097803,-0.999728969781972,-0.00397704210262544,-14.2008917453338;-0.0145288236957037,-0.00431806690482854,0.999896817128150,6.31722374408329;0,0,0,1];

%Checkerboard Calibration (16-03-29 dataset)
%H = [-0.954419965690634,-0.0149980864223171,-0.00379733161617120,1220.91072044943;0.0150371395688773,-0.954375618521059,-0.0100681159932028,-7.44561245393950;-0.00363749907742485,-0.0101271785083495,0.954485572222311,-4.41230872199416;0,0,0,0.954545454545455];

%Checkerboard Calibration (16-04-01 dataset)
%H = [-0.999596625580790,-0.0173226503674928,-0.0225627752976572,1281.49693639937;0.0174579501825619,-0.999830278842096,-0.00582929619919913,-10.7344705225465;-0.0224603607725686,-0.00622172238024872,0.999729203837576,12.2808819453738;0,0,0,1];

%Checkerboard Calibration (14-04-28 dataset) "SET B"
%H = [-0.999658079026256,-0.0199718904105454,-0.0168313278643821,1277.92985696697;0.0200011848509301,-0.999798417377584,-0.00158835611108141,-11.5311333787936;-0.0167976883589561,-0.00192378298985459,0.999856078205371,7.10304214143702;0,0,0,1];

%ARToolkit - Hiro - 16_04_26 - with 14_04_28 checkerboard handeye
%H = [-0.999796022739253,-0.0216404091181083,-0.00426861770571225,1274.74267163451;0.0215909905616854,-0.999721078332015,0.00802670443313294,-15.5346130190007;-0.00446442448684792,0.00795672442454891,0.999973081143292,-1.47724623613463;0,0,0,1];

%ARToolkit - Hiro - 16_04_26 - with ATK Hiro handeye
%H = [-0.999924841125299,-0.0132869607177370,-0.0101643614929748,1276.51525559459;0.0133312979907460,-0.999896893894731,-0.00211540469901311,-8.65838442450392;-0.0101415965370624,-0.00223986680321609,0.999998157696351,-0.695400210548762;0,0,0,1];

%ARToolkit - 4x3 - 16_04_26 - with 14_04_28 checkerboard handeye
%H = [-0.999515624406612,-0.0233882548296717,-0.0171655904514067,1280.13996491622;0.0231926345627694,-0.999682115342244,0.00830684208897645,-22.5766524032988;-0.0173469709934338,0.00792221352219293,0.999762162494616,5.19323049170579;0,0,0,1];

%ARToolkit - 4x3 - 16_04_26 - with ATK 4x3 handeye
H = [-0.999485661032632,-0.0264731423861542,-0.0179289332490419,1277.51546246639;0.0265577415925886,-0.999649452522345,-0.00383374853019225,-15.2586758701959;-0.0178281949566318,-0.00430282872433034,0.999839086376437,7.79799356225630;0,0,0,1];

%ARToolkit - 3corner - 14_04_28 - with 14_04_28 checkerboard handeye
%H = [-0.999540965604073,-0.00126262407854438,-0.0303360142745566,1282.36762711811;0.00149268534066999,-0.999982821205488,-0.00793962194001659,4.93398534052230;-0.0303162904557580,-0.00797167467941478,0.999511259333277,16.8583039832587;0,0,0,1];

%ARToolkit - 3corner - 14_04_28 - with ATK 3corner handeye
%H = [-0.999903701213963,-0.0139999157026888,-0.00215710368541512,1275.33790127138;0.0140110456118105,-0.999904783098902,-0.00428828253067677,-11.8852050902765;-0.00210557176963909,-0.00431906027312873,0.999998684499554,-2.28381139698969;0,0,0,1];

%RGB-D Calibration
%H = [-0.997520337302928,-0.0337548467849146,0.00668894500755426,1269.42583274738;0.0344062005247205,-0.991171111405880,0.125873913546245,-65.1727762887343;0.00238935824715405,0.125905284215395,0.990279963307384,-9.81098427880939;0,0,0,1];

%% plot board + bots
if graph_it
    Draw_Bots(H)
    xlabel('x (mm)')
    ylabel('y (mm)')
    zlabel('z (mm)')
end

%% Compute and Plot Error Diagram
err = zeros(1,size(Jerry_Pose,1));
err_vec = zeros(4,size(Jerry_Pose,1));
for i = 1:size(Jerry_Pose,1)
    
    %Jerry Pose
    Ej = Jerry_Pose(i,4:6) * pi/180;
    %Rj = eul2rotm(Ej);
    Rj = eye(3);
    tj = (Jerry_Pose(i,1:3))';
    Jerry_H = [Rj tj; 0 0 0 1];

    %Dash Pose
    Ed = Dash_Pose(i,4:6) * pi/180;
    Rd = eul2rotm(Ed);
    td = (Dash_Pose(i,1:3))';
    Dash_H = [Rd td; 0 0 0 1];

    %Dash Pointer in robot 1 (Jerry) frame
    Ptr_D_b1 = H * Dash_H * needleDash;

    %Jerry Pointer in robot 1 (Jerry) frame
    Ptr_J_b1 = Jerry_H * needleJerry;

    %Compute Error
    err(i) = norm( Ptr_D_b1 - Ptr_J_b1 );
    err_vec(:,i) = Ptr_D_b1 - Ptr_J_b1;
    
    if graph_it
        %select plot color by error ammount
        if err(i) < 2
            color = 'g';
        elseif err(i) < 3
            color = 'b';
        elseif err(i) < 4
            color = 'm';
        else
            color = 'r';
        end

        %print graph
        plot3(Ptr_D_b1(1),Ptr_D_b1(2),Ptr_D_b1(3),strcat(color,'o'))
        plot3(Ptr_J_b1(1),Ptr_J_b1(2),Ptr_J_b1(3),strcat(color,'x'))
        
        if scaled_vectors
            %print scaled vector
            sc_v = Ptr_J_b1 + vec_scale * (Ptr_D_b1 - Ptr_J_b1);
            plot3([Ptr_J_b1(1) sc_v(1)], [Ptr_J_b1(2) sc_v(2)], [Ptr_J_b1(3) sc_v(3)],color)
        end
    end
    
end
avg_err = mean(err);
   
%%

%procusties for ground truth transformation
JP = Jerry_Pose(:,1:3);
DP = Dash_Pose(:,1:3);

[d,Z,tr] = procrustes(JP,DP,'scaling',false,'reflection',false);
Hp = [tr.T, tr.c(1,:)'; 0 0 0 1]; %unpack into homogenous matrix

H_t = H(1:3,4);
Hp_t = Hp(1:3,4);
H_R = H(1:3,1:3);
Hp_R = Hp(1:3,1:3);

tr_err_t = norm(H_t - Hp_t);
R_err = H_R * inv(Hp_R);        %compare rotation matrices, should be ~I
R_err_aa = vrrotmat2vec(R_err); %convert to axis angle representation
tr_err_alpha = R_err_aa(4);     %take anlge of error

disp('average linear error of sampled points (mm)')
disp(avg_err)
disp('transformation linear error (mm)')
disp(tr_err_t)
disp('transformation rotation error (rad)')
disp(tr_err_alpha)
disp('transformation rotation error (deg)')
disp(tr_err_alpha * 180 / pi)

%%
if graph_it && spin_it
    spin
end