// -------------------------------------------------------------------------------------------
// Basic Master
// -------------------------------------------------------------------------------------------
//
//
// -------------------------------------------------------------------------------------------

#include <i2c_t3.h>
#include <ADC.h>

// These constants won't change:
// Buttons
const int button1 = 5;
const int button2 = 8;
const int button3 = 11;
const int button4 = 24;
const int button5 = 27;
const int button6 = 30;

// 8 amplifier channels max per multiplex
const int channels = 1;
//const int sensorPin0 = A0;
//const int sensorPin1 = A17;
//const int sensorPin2 = A15;
//const int sensorPin3 = A14;
//const int sensorPins[] = {sensorPin0,sensorPin1,sensorPin2,sensorPin3};
//const int sensorPins[] = {A0};
//const int sensorPins[] = {A0,A17,A15,A14};
const int sensorPins[] = {A0,A1,A2,A3,A4,A5,A6,A7,A8,A9,A12,A13,A14,A16,A17,A18,A19,A20,A21,A22};

const int ledPin_calib = 0;        // Calibration process pin
const int ledPin_adc = 1;        //  ADC process pin

// Loop conditions
bool isRead;
bool isTempADC;
bool isPressing;
bool isSync;

// Memory
uint8_t currAddr;
uint8_t currPlex;
uint16_t rawTwoByte;
uint8_t writeArray[2];
uint8_t targetI2C = 0x40; // target Slave address of PGA309 (per devices)
uint8_t targetPlex = 0x70; // target Slave address of Multiplex
uint8_t testCtrl = 0x01; // Initial channel enable of multiplex
uint8_t byte1; uint8_t byte2;

// Data array initialization (try to control for 1kHz/channel)
const unsigned long bufferAmt = 20000;
//const unsigned long timeDiv = channels;
const unsigned long timeDiv = 20;
const unsigned long timeAmt = bufferAmt / timeDiv;
int bufferPacket[bufferAmt];
long timePacket[timeAmt];

ADC *adc = new ADC();

void setup()
{
  // Set adc pins to input
  //pinMode(sensorPin0,INPUT);
  //pinMode(sensorPin1,INPUT);
  //pinMode(sensorPin2,INPUT);
  //pinMode(sensorPin3,INPUT);
  
  pinMode(A0,INPUT);
  pinMode(A1,INPUT);
  pinMode(A2,INPUT);
  pinMode(A3,INPUT);
  pinMode(A4,INPUT);
  pinMode(A5,INPUT);
  pinMode(A6,INPUT);
  pinMode(A7,INPUT);
  pinMode(A8,INPUT);
  pinMode(A9,INPUT);
  pinMode(A12,INPUT);
  pinMode(A13,INPUT);
  pinMode(A14,INPUT);
  pinMode(A15,INPUT);
  pinMode(A16,INPUT);
  pinMode(A17,INPUT);
  pinMode(A18,INPUT);
  pinMode(A19,INPUT);
  pinMode(A20,INPUT);
  pinMode(A21,INPUT);
  pinMode(A22,INPUT);

  // Set led pins to output
  pinMode(LED_BUILTIN,OUTPUT);    // LED
  pinMode(ledPin_calib,OUTPUT);    // LED
  pinMode(ledPin_adc,OUTPUT);    // LED
  digitalWrite(LED_BUILTIN,LOW);  // LED off
  digitalWrite(ledPin_calib,LOW);    // LED off
  digitalWrite(ledPin_adc,LOW);    // LED off

  /*
  // Multiplexer has slave channel 0x70 (if all 3 low bits set to 0)
  pinMode(addrPin0,OUTPUT);       // Control for Send/Receive select
  pinMode(addrPin1,OUTPUT);       // Control for address
  pinMode(addrPin2,OUTPUT);       // Start address program
  digitalWrite(addrPin0,LOW);    // LED off
  digitalWrite(addrPin1,LOW);    // LED off
  digitalWrite(addrPin2,LOW);    // LED off
  */
  
  // internal resistor enable for 
  pinMode(button1,INPUT_PULLUP);       // Control for Send/Receive select
  pinMode(button2,INPUT_PULLUP);       // Control for address
  pinMode(button3,INPUT_PULLUP);       // Start address program
  pinMode(button4,INPUT_PULLUP);       // Send ADC readings
  pinMode(button5,INPUT_PULLUP);       // Calibrate
  pinMode(button6,INPUT_PULLUP);       // Switch plex channel

  //8,10,12,16
  // Keep at 16 bit, takes up same amount of memory
  adc->setResolution(16);
  //0,4,8,16,32
  adc->setAveraging(16);
  adc->setSamplingSpeed(ADC_HIGH_SPEED);
  adc->setConversionSpeed(ADC_HIGH_SPEED);
  
  // Setup for Master mode, pins 18/19, external pullups, 400kHz, 200ms default timeout, Teensy sets 0x00 address, but does not matter as master
  Wire.begin(I2C_MASTER, 0x00, I2C_PINS_18_19, I2C_PULLUP_EXT, 400000);
  Wire.setDefaultTimeout(200000); // 200ms

  // Initial loop conditions
  isPressing = false;
  isRead = false;
  isTempADC = true;
  isSync = true;

  // Initialize memory addresses
  currAddr = 0x00;
  currPlex = 0x01;
  byte1 = 0x00;
  byte2 = 0x00;

  switchPlex(currPlex);

  Serial.begin(115200);
  delay(100);
}


// Incorporate button interrupts?
void loop(){

    // Press button 1; switch i2c intent b/t read and write
    digitalWrite(LED_BUILTIN,LOW);  // LED on
    if(digitalRead(button1) == LOW){
      isPressing = true;
      isRead = !isRead;
      Serial.println("Read/Write state switched: ");
      if(isRead)
      Serial.println("Set to Read");
      else
      Serial.println("Set to Write");
      Serial.println(" ");
      delay(800);
    }

    // Press button 2; switch i2c address intent from 0 to 8
    if(digitalRead(button2) == LOW){
      isPressing = true;
      if(currAddr == 0x08)
        currAddr = 0x00;
      else
        currAddr = currAddr + 1;
      Serial.println("Context switched: ");

      switch(currAddr) {
        case 0x00:
        Serial.println("Temp ADC Read");
        isTempADC = true;
        break;         
        case 0x01:
        Serial.println("Fine Offset Adjust");
        isTempADC = false;
        break; 
        case 0x02:
        Serial.println("Fine Gain Adjust");
        break;
        case 0x03:
        Serial.println("Reference Control and Linearization");
        break;
        case 0x04:
        Serial.println("PGA Coarse Offset & Gain, Output Amp Gain");
        break;
        case 0x05:
        Serial.println("PGA Configuration and Over/Under-Scale");
        break;    
        case 0x06:
        Serial.println("Temp ADC Control");
        break;
        case 0x07:
        Serial.println("Output Enable Counter Control");
        break;
        case 0x08:
        Serial.println("Fault Monitor Output");
        break; 
        default:
        Serial.println("Read status unknown");
      }
      Serial.println(" ");
      delay(800);
    }

    // Press button 3; start test routine for writing/reading PGA register
    if(digitalRead(button3) == LOW){
      isPressing = true;
      digitalWrite(LED_BUILTIN,HIGH);   // LED on
      // Transmit to Slave
      Serial.println("Message creation started.");
      Serial.printf("Current PGA device is 0x%X. \n",currPlex);
      Serial.printf("Current PGA address is 0x%X. \n",currAddr);
      // Initialize Tx buffer
      if(isRead){
        readFromPGA();
      } else if (!isRead && (currAddr != 0x00) && (currAddr != 0x08)) {
        writeToPGA();
      } else {
        Serial.println("Can't write to read-only addr.");
      }
      Serial.println(" ");

      // Check if communication failed
      print_status();
      print_errors();
      
      digitalWrite(LED_BUILTIN,LOW);    // LED off
      delay(800);                       // Delay to space out tests      
    }
    
    // Button 4; start calibration routine
    if(digitalRead(button4) == LOW){
      calibration();
      delay(100);
    }

    // Button 5; start test routine for recording ADC and sending over serial
    if(digitalRead(button5) == LOW){
      digitalWrite(ledPin_adc,HIGH);    // LED off
      Serial.println("ADC data being sent to MatLab.");

      //sendADCData(sensorPins[0]);
      sendADCData();
      
      Serial.println("ADC data transmission done.");
      digitalWrite(ledPin_adc,LOW);    // LED off
      delay(100);
    }
    // Button 6; manually switch multiplex switch
    if(digitalRead(button6) == LOW){

      if((currPlex << 1) > (0x01 << (channels - 1))){
        currPlex = 0x01;
      } else {
        currPlex = currPlex << 1;
      }
      
      Serial.println("Manual multiplex change.");
      switchPlex(currPlex);
      Serial.println("Change completed done.");
      delay(800);
    }
}

void readADCData(){
  Serial.println("Reading ADC...");
  unsigned long j = 0;
  int adcReading;
  clearADCData();
  unsigned long startTime = millis();
  // Record for [duration] ms
  // Causes memory addresses (currPlex, currAddr, etc.) to be corrupted
  while(j < bufferAmt){
    if ( channels == 2 ) {
      
      ADC::Sync_result result = adc->analogSyncRead(sensorPins[0],sensorPins[1]);
      bufferPacket[j] = result.result_adc0;
      bufferPacket[j+1] = result.result_adc1;
      
    } else if ( channels == 1 ){
      
      adcReading = adc->analogRead(sensorPins[0]);
      bufferPacket[j] = adcReading;
      // Record timestamp when returning to first channel in us
      
    } else {   
      int currChan = j%channels;
      adcReading = adc->analogRead(sensorPins[currChan]);
      //Serial.printf("Channel %d has been read.\n",currChan);
      bufferPacket[j] = adcReading;
    }
    
    if (j % timeDiv == 0){
      timePacket[j / timeDiv] = micros() - (startTime * 1000);
    }
    
    if (j != bufferAmt) {
      if ( channels == 2 ) {
        j = j + 2;     
      } else if ( channels == 1 ){
        j = j + 1;
      } else {
        j = j + 1;
      }
    }
    
  }
  Serial.printf("Buffer has been filled with %d entries \n", j);
  unsigned long elapsed = millis() - startTime;
  Serial.printf("The sampling time was %d ms. \n",elapsed);
  float sampFreq = ((float) j) / elapsed;
  Serial.printf("Sampling frequency was %4.4f kHz. \n",sampFreq);
}

// Send data through USB
void sendADCData(){
  readADCData();
  Serial.println("Sending ADC...");
  unsigned long startTime = millis();
  unsigned long duration = 1000; // 1 sec
  while(millis() < (startTime + 2*duration));
  long currTime = 0;
  for (unsigned long k=0;k<bufferAmt;k++){
    //String msg = "0 ";
    String msg = "33100 ";
    if (k % timeDiv == 0){
      currTime = timePacket[k / timeDiv]; 
    }
    //msg += currTime;
    //msg += " ";
    msg += bufferPacket[k];
    msg += " 33130";
    Serial.println(msg);
  }
  
}

// Remove current entries from data array
void clearADCData(){
  Serial.println("Deleting current entries...");
  for (unsigned long k=0;k<bufferAmt;k++){
    bufferPacket[k] = 0;
    if (k % timeDiv == 0){
      timePacket[k / timeDiv] = 0; 
    }
  }
  Serial.println("Deletion complete.");
}

// Find average entry in data array
float averageADCData(){
  unsigned long total = 0;
  for (unsigned long k=0;k<bufferAmt;k++){
    total = total + bufferPacket[k];
  }
  float average = total / bufferAmt;
  return average;
}

// Reads a channel per second, finds average voltage value in mV, adjusts offset register with compensation value
void calibration(){
  uint8_t i2cAddr = 0x01;
  float Vcc1 = 3.3;
  for (int i = 0; i < channels; i++){
    Serial.printf("Channel %d is now selected. \n",i+1);
    
    readADCData();
    
    float avg = averageADCData();
    Serial.printf("Average ADC level is %4.2f. \n",avg);
    // Assume max level is 2^16 (int type)
    float adjust_mV = (avg / 65536) * 1000 * Vcc1;
    Serial.printf("Average voltage in mV is %4.2f. \n",adjust_mV);
    // change to fine offset, to half way point?
    uint16_t offsetMsg = PGAOffsetGain(0, 0, -adjust_mV);
    Serial.printf("Corresponding hex is 0x%X. \n",offsetMsg);
    splitTwoByte(writeArray,offsetMsg);
    
    Serial.println("#####");
    
    // Start message
    switchPlex(i2cAddr);

    Serial.println("#####");
    
    Serial.println("Sending to PGA.");
    Wire.beginTransmission(targetI2C);
    Wire.write(0x04);
    Wire.write(writeArray,2); // transmits two bytes from an array
    Wire.endTransmission();
    
    // Check if communication failed
    print_status();
    print_errors();
    
    i2cAddr = i2cAddr << 1;
    Serial.println("Channel being switched.");
    Serial.println("-------------------------------------------");
  }
}

void print_status()
{
  Serial.println("Status message:");
  switch(Wire.status())
  {
  case I2C_WAITING:
      Serial.println("Waiting for action/cmd.");
      break;
  case I2C_SENDING: 
      Serial.println("Sending input.");
      break;
  case I2C_SEND_ADDR: 
      Serial.println("Sending address.");
      break;
  case I2C_RECEIVING: 
      Serial.println("Receiving data.");
      break;
  case I2C_TIMEOUT: 
      Serial.println("Timeout has occurred.");
      break;
  case I2C_ADDR_NAK: 
      Serial.println("No acknowledge bit for address action.");
      break;
  case I2C_DATA_NAK: 
      Serial.println("No acknowledge bit for data action.");
      break;
  case I2C_ARB_LOST: 
      Serial.println("ARB_LOST status.");
      break;
  case I2C_SLAVE_TX: 
      Serial.println("Transmitting as a slave.");
      break;
  case I2C_SLAVE_RX: 
      Serial.println("Receiving as a slave.");
      break;
  default:
      Serial.println("Unknown error.");
      break;
  }
  Serial.println(" ");
}

void print_errors()
{
  Serial.println("Error message:");
  switch(Wire.getError())
  {
  case 0:
      Serial.println("No errors found.");
      break;
  case 1: 
      Serial.println("Data too long.");
      break;
  case 2:
      Serial.println("Received no acknowledge bit from address action.");
      break;
  case 3:
      Serial.println("Received no acknowledge bit from data action.");
      break;
  case 4:
      Serial.println("Other error has occurred.");
      break;
  default:
      Serial.println("Unknown error.");
      break;
  }
  Serial.println(" ");
}

void readFromPGA(){
  Serial.println("Reading from slave. ");
  Serial.printf("Read address is 0x%X.\n", currAddr);
  Wire.beginTransmission(targetI2C);   // Slave address
  Wire.write(currAddr);
  // Read from Slave
  Wire.requestFrom(targetI2C, 2, I2C_NOSTOP);
  Serial.printf("There are %d bytes available from PGA. \n", Wire.available());
  while(Wire.available()){
    //Wire.read()
    byte2 = Wire.readByte();
    byte1 = Wire.readByte();
  }
  // Temperature reading
  if(isTempADC){
    float temp = tempConvert(byte1,byte2);
    Serial.printf("The current temperature is %f degrees Celsius. \n", temp);
  // Error code reading
  } else {
    uint16_t errorCode = concatBytes(byte1,byte2);
    Serial.printf("The fault code is %d. \n", errorCode);
  }
  // Transmit Tx buffer (unneccessary for reading?)
  Wire.endTransmission();           // Transmit to Slave
}

void writeToPGA(){
  // Print message to register to modify amp behavior
  Serial.println("Sending to slave.");
  Wire.beginTransmission(targetI2C);   // Slave address
  Wire.write(currAddr);
  Serial.printf("Write address is 0x%X.\n", currAddr);
  // Has to be two addresses
  // Address 1-8, then address 9-16
  rawTwoByte = writeSelect(currAddr);
  splitTwoByte(writeArray,rawTwoByte);
  Wire.write(writeArray,2); // transmits two bytes from an array
  // Transmit Tx buffer
  Wire.endTransmission();           // Transmit to Slave
  Serial.printf("Raw write data is 0x%X.\n", rawTwoByte);
  Serial.printf("First byte sent is 0x%X.\n", writeArray[1]);
  Serial.printf("Second byte sent is 0x%X.\n", writeArray[0]);
}

void switchPlex(uint8_t ctrlReg){
  uint8_t regState = 0x00;
  Serial.println("Sending to multiplex.");
  //Stops here when channels = 1;
  Wire.beginTransmission(targetPlex);
  Serial.printf("Intended register state is 0x%X.\n", ctrlReg);
  Wire.write(ctrlReg);
  Wire.endTransmission();

  // Check if communication failed
  print_status();
  print_errors();

  Serial.println("Verifying multiplex register.");
  Wire.requestFrom(targetPlex, 1, I2C_NOSTOP);
  while(Wire.available()){
    regState = Wire.readByte();
  }

  // Check if communication failed
  print_status();
  print_errors();
  
  Serial.printf("New register state is 0x%X.\n", regState);
  Serial.println(" ");
}

// Converts ADC temperature readings into usable float data
float tempConvert(uint8_t byte1, uint8_t byte2){
  // Concat the two bytes
  uint16_t count = concatBytes(byte1,byte2);
  // Multiply the 2-byte by the multiplier to yield Celsius reading
  float tempC = 0.0625 * count;
  return tempC;
}

uint16_t concatBytes(uint8_t byte1, uint8_t byte2){
  uint16_t twoByte;
  twoByte = byte1;
  twoByte = twoByte<<8;
  twoByte |= byte2;
  Serial.printf("The raw read data is 0x%X. \n", twoByte);
  return twoByte;
}

void splitTwoByte(uint8_t *pWrite, uint16_t twoByte){
  // Clear array
  pWrite[1] = 0x00;
  pWrite[0] = 0x00;  
  uint8_t botByte = (uint8_t) (twoByte & 0xFF);
  uint8_t topByte = (uint8_t) ((twoByte >> 8) & 0xFF);
  pWrite[1] = topByte;
  pWrite[0] = botByte;
}

// Selects an internal register to write to
uint16_t writeSelect(uint8_t currAddr){
  uint16_t writeMsg;
  switch(currAddr) {
    case 0x01:
    writeMsg = fineOffsetAdjust();
    break; 
    case 0x02:
    writeMsg = fineGainAdjust();
    break;
    case 0x03:
    writeMsg = refCtrlLinear();
    break;
    case 0x04:
    //writeMsg = PGAOffsetGain(0,0,0);
    //writeMsg = PGAOffsetGain(4,2,-15.0);
    //writeMsg = PGAOffsetGain(8,2,0);
    //writeMsg = PGAOffsetGain(4,3,0);
    //writeMsg = PGAOffsetGain(16,2,0);
    writeMsg = PGAOffsetGain(4,2.4,0);
    break;
    case 0x05:
    writeMsg = PGAConfigLimit();
    break;    
    case 0x06:
    writeMsg = TempADCCtrl();
    break;
    case 0x07:
    writeMsg = OutEnableCounterCtrl();
    break;
    default:
    writeMsg = fineOffsetAdjust();
  }
  return writeMsg;
}

uint16_t fineOffsetAdjust(){
  // Assume 5 volt reference
  float V_ref = 3.3;
  //float V_zero = 0.25*V_ref;
  float V_zero = 0.5*V_ref;
  float ZD_float = V_zero * 65535 / V_ref;
  uint16_t ZD = (uint16_t) ZD_float;
  return ZD;
}

uint16_t fineGainAdjust(){
  //float targetG = 0.5;
  //float targetG = 0.33;
  float targetG = 1;
  float GD_float = (targetG - 0.33333333) * 1.5 * 65536;
  uint16_t GD = (uint16_t) GD_float;
  return GD;
}

uint16_t refCtrlLinear(){
  uint16_t result;

  uint8_t RFB = 0x00;

  // Linear Adjust, V_exc gain select
  // 0x00 : -0.166 V_fb to +0.166 V_fb lineariz. DAC; 0.83 V_ref of V_exc gain
  // 0x01 : -0.124 V_fb to +0.124 V_fb lineariz. DAC; 0.52 V_ref of V_exc gain
  uint8_t EXS = 0x00;

  // Enable V_exc
  uint8_t EXEN = 0x00;

  // Select internal V_ref (0-4.096V, 1-2.5V)
  uint8_t RS = 0x00;

  // Enable internal V_ref
  uint8_t REN = 0x00;

  float FSR;
  if(EXS == 0x01){
    FSR = 0.166;
  }else{
    FSR = 0.124;
  }
  
  float desiredFBRatio = 0.04183; // ratio
  int8_t LD = (int8_t) (desiredFBRatio / (FSR / 127));

  // Starts with 4 bits
  result = RFB; result = result<<1;
  result |= EXS; result = result<<1;
  result |= EXEN; result = result<<1;
  result |= RS; result = result<<1;
  result |= REN; result = result<<8;
  result |= LD;
  
  return result;
}

uint16_t PGAOffsetGain(float dGainF, float dGainO, float dOffset){
  // dOffset should be in mV 
  uint16_t result;
  
  uint8_t OWD = 0x01; // Disable PRG pin

  uint8_t GO, GI;
  
  // Output Amp gain table
  // 0x00 - 2
  // 0x01 - 2.4
  // 0x02 - 3
  // 0x03 - 3.6
  // 0x04 - 4.5
  // 0x05 - 6
  // 0x06 - 9
  // 0x07 - Disable internal feedback

  if (dGainO >= 9){
    GO = 0x06;
  } else if (dGainO >= 6){
    GO = 0x05;
  } else if (dGainO >= 4.5){
    GO = 0x04;
  } else if (dGainO >= 3.6){
    GO = 0x03;
  } else if (dGainO >= 3){
    GO = 0x02;
  } else if (dGainO >= 2.4){
    GO = 0x01;
  } else {
    GO = 0x00; // Gain of 2
  } 

  // 0x00 - V_in1 = V_inp; V_in2 = V_inn;
  // 0x01 - V_in1 = V_inn; V_in1 = V_inp;
  uint8_t GI3 = 0x00;

  // Frontend gain table
  // 0x00 - 4
  // 0x01 - 8
  // 0x02 - 16
  // 0x03 - 23.27
  // 0x04 - 32
  // 0x05 - 42.67
  // 0x06 - 64
  // 0x07 - 128

  if (dGainF >= 128){
    GI = 0x07;
  } else if (dGainF >= 64){
    GI = 0x06;
  } else if (dGainF >= 42.67){
    GI = 0x05;
  } else if (dGainF >= 32){
    GI = 0x04;
  } else if (dGainF >= 23.27){
    GI = 0x03;
  } else if (dGainF >= 16){
    GI = 0x02;
  } else if (dGainF >= 8){
    GI = 0x01;
  } else {
    GI = 0x00; // Gain is 4
  } 
  
  uint8_t RFB = 0x00;

  // Assume 3.3 volt reference
  float V_ref = 3.3; // V
  uint8_t OS = abs(dOffset) / (V_ref * 0.85);

  uint8_t OS5 = 0x00;
  
  if (dOffset < 0){
    OS5 = 0x01;
  }

  // Limits OS register for magnitude, prevents overflow into negative offset and other registers
  if (OS > 0x0F){
    OS = 0x0F;
  }
  
 
  // Starts with 1 bits
  result = OWD; result = result<<3;
  result |= GO; result = result<<1;
  result |= GI3; result = result<<3;
  result |= GI; result = result<<3;
  result |= RFB; result = result<<1;
  result |= OS5; result = result<<4;
  result |= OS;
  
  return result;
}

uint16_t PGAConfigLimit(){
  uint16_t result;
  
  uint8_t RFB = 0x00;

  // Table 6-11 in User manual
  uint8_t CLK_CFG = 0x00;

  // Enable ext fault comp group
  uint8_t EXTEN = 0x00;

  // Enable int fault comp group
  uint8_t INTEN = 0x00;

  // V_out polarity (1-high, 0-low) when ext fault detected
  uint8_t EXTPOL = 0x00;

  // V_out polarity (1-high, 0-low) when int fault detected
  uint8_t INTPOL = 0x00;

  // Over/under-scale Limit Enable
  uint8_t OUEN = 0x00;
  
  // 0x00 - 0.9708 V_ref
  // 0x01 - 0.9610 V_ref
  // 0x02 - 0.9394 V_ref
  // 0x03 - 0.9160 V_ref
  // 0x04 - 0.9102 V_ref
  // 0x05 - 0.7324 V_ref
  // 0x06 - 0.5528 V_ref
  // 0x07 - Reserved
  uint8_t HL = 0x00;

  // 0x00 - 0.02540 V_ref
  // 0x01 - 0.02930 V_ref
  // 0x02 - 0.03516 V_ref
  // 0x03 - 0.03906 V_ref
  // 0x04 - 0.04492 V_ref
  // 0x05 - 0.05078 V_ref
  // 0x06 - 0.05468 V_ref
  // 0x07 - 0.06054 V_ref
  uint8_t LL = 0x00;

  // Starts with 2 bits
  result = RFB; result = result<<2;
  result |= CLK_CFG; result = result<<1;
  result |= EXTEN; result = result<<1;
  result |= INTEN; result = result<<1;
  result |= EXTPOL; result = result<<1;
  result |= INTPOL; result = result<<1;
  result |= RFB; result = result<<1;
  result |= OUEN; result = result<<3;
  result |= HL; result = result<<3;
  result |= LL;

  return result;
}

uint16_t TempADCCtrl(){
  uint16_t result;
  
  uint8_t RFB = 0x00;
  
  // Enable twice conversion speed for temp ADC
  uint8_t ADC2X = 0x00;
  
  // Start or restart the ADC
  uint8_t ADCS = 0x01;
  
  // Enable 7microA TEMP_in current source (I_temp)
  uint8_t ISEN = 0x00;
  
  // Enable continuous conversion
  uint8_t CEN = 0x00;
  
  // Enable Internal temp mode
  // Set ADC2X to 0; AREN to 0; RV, M, G to 00;
  uint8_t TEN = 0x00;
  
  // Enable internal 2.048 V reference, ignores RV
  uint8_t AREN = 0x00;
  
  // External temp ref
  // 0x00 - V_ref
  // 0x01 - V_exc
  // 0x02 - V_SA
  // 0x03 - Reserved
  uint8_t RV = 0x00;
  
  // Temp ADC Mux Select
  // 0x00 - TEMP_in (+), GND_a (-)
  // 0x01 - V_exc (+), TEMP_in (-)
  // 0x02 - V_out (+), GND_a (-)
  // 0x03 - V_ref (+), TEMP_in (-) 
  uint8_t MUX = 0x00;
  
  // Temp ADC PGA Gain
  // 0x00 - 1
  // 0x01 - 2
  // 0x02 - 4
  // 0x03 - 8 
  uint8_t GAIN = 0x00;

  // Refer to Table 6-17 in PGA309 User Manual (it's too long to comment O_o)
  uint8_t RES = 0x00;

  // Starts with 2 bits
  result = RFB; result = result<<1;
  result |= ADC2X; result = result<<1;
  result |= ADCS; result = result<<1;
  result |= ISEN; result = result<<1;
  result |= CEN; result = result<<1;
  result |= TEN; result = result<<1;
  result |= AREN; result = result<<2;
  result |= RV; result = result<<2;
  result |= MUX; result = result<<2;
  result |= GAIN; result = result<<2;
  result |= RES;
  
  return result;
}

uint16_t OutEnableCounterCtrl(){
  uint16_t result;
  
  uint8_t RFB = 0x00;
  
  // Temp ADC Delay (OEN decimal value * 14) ms
  int tempADCDelay = 98; // ms
  uint8_t DLY = tempADCDelay / 14;
  
  // V_out enable timeout
  int desiredTimeout = 2240; // ms
  uint8_t OEN = desiredTimeout / 14;

  // Starts with 4 bits
  result = RFB; result = result<<4;
  result |= DLY; result = result<<8;
  result |= OEN;  
  
  return result;
}

