import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import scipy.stats as stats
import datetime
from segment import segment_bytime
from processdata import get_lengthofstay

''' This function uses the segment and length of stay functions in order to get a list of durations.
	This list has the durations of stay in 4P for the chosen number of patients.
    Inputs: 
    simulation_date- the current date and time from the simulation
    start_range- the beginning of the range to segment the time of arrival, should be before datetime
    end_ range- the end of hte range to segment time of arrival, should be after datetime
    Outputs:
    duration_list- a list of durations whose size indicates the number of patients to introduce
'''
def get_duration(simulation_date, start_range, end_range):
	# Uses the general ADTs dataframe- this is all of the relevant historical data
	ADTs = pd.read_pickle('ADTs.pkl')

	# Gets the ADTs of the relevant patients given the temporal criteria and the number of patients 
	# to be introduced
	# Accounts for ranges that span overnight
	if end_range < start_range:
		simulation_date1 = simulation_date - datetime.timedelta(days=1)
		Seg_ADTs1, pat_num1 = segment_bytime(ADTs,simulation_date1,start_range,24)
		Seg_ADTs2, pat_num2 = segment_bytime(ADTs,simulation_date,0,end_range)
		Seg_ADTs = pd.concat([Seg_ADTs1, Seg_ADTs2])
		pat_num = pat_num1 + pat_num2
	else:
		Seg_ADTs, pat_num = segment_bytime(ADTs,simulation_date,start_range,end_range)

	# Creates list to hold durations of 4P patients
	duration_list = get_lengthofstay(Seg_ADTs, pat_num)

	return duration_list

''' This function uses a lookup table of historical information to get the durations of the
	correct number of patients to enter the unit at the given simulation time.
    Inputs: 
    simulation_date- the current date and time from the simulation
    start_range- 	 the beginning of the range to segment the time of arrival
    end_ range- 	 the end of the range to segment time of arrival
    lookup-          a 4x7x4 table that contains duration lists and number of patients per hour
    				 probability distribution for each time partition
    Outputs:
    samples
'''
def get_duration_new(simulation_date, start_range, end_range,lookup):
	# Determine the temporal parameters given the simulation_date and time range values
	month = simulation_date.month
	if month in [12,1,2]:
		season = 0
	if month in [3,4,5]:
		season = 1
	if month in [6,7,8]:
		season = 2
	if month in [9,10,11]:
		season = 3
	
	dayofweek = simulation_date.weekday()

	if start_range == 7:
		time_range = 0
	elif start_range == 15:
		time_range = 1
	elif start_range == 19:
		time_range = 2
	else:
		time_range = 3

	# Sample from the probability distribution to determine the number of patients to enter
	# at the given hour
	pat_probs = lookup[season][dayofweek][time_range][1]
	patient_number = np.random.choice(a = len(pat_probs), p=pat_probs)

	# Obtains duration values if patients are going to enter the unit
	if patient_number > 0:
		duration_list = lookup[season][dayofweek][time_range][0]

		# Fits random variable to and samples from the duration distribution
		plt.hist(duration_list, normed=True, bins=100)
		xt = plt.xticks()[0]  
		xmin, xmax = min(xt), max(xt)  
		lnspc = np.linspace(xmin, xmax, len(duration_list))
		hist = np.histogram(duration_list, bins=100)
		hist_dist = stats.rv_histogram(hist)
		plt.plot(lnspc, hist_dist.pdf(lnspc), label='PDF')

		samples = hist_dist.rvs(size = patient_number)
	else:
		samples = []

	# Return duration list of correct size
	return samples
