import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import datetime

''' This function segments ADTs according to temporal attributes like season, day of week, and time of arrival.
	It also chooses the number of patients to be simulated probabilistically based on historical data.
    Inputs: 
    dataframe_in- pandas dataframe of all ADTs
    curr_datetime- current simulation time
    start_range- the beginning of the range to segment the time of arrival, should be before datetime
    end_ range- the end of hte range to segment time of arrival, should be after datetime
    Outputs:
    Return_ADTs- a dataframe of ADT information only for patients who meet temporal segmenting criteria
    patient_number- the number of patients to be added to the simulation at this hour
'''
def segment_bytime(dataframe_in,curr_datetime,start_range,end_range):
	ADTs = dataframe_in

	# Obtains a list of the visitIDs from the ADT dataframe
	UniqueIDs = ADTs.VisitID.unique()

	# Gets the current month and day of week of the simulation
	month = curr_datetime.month
	dayofweek = curr_datetime.weekday()

	# Creates a dataframe of ADT information whose department is 4P
	FourPs_Only = ADTs[ ADTs['ADT_DEPARTMENT_NAME'].isin(['HCGH 4P ACUTE']) ]
	# Sorts the 4P dataframe by time
	FourPs_Only = FourPs_Only.sort_values('IN_DTTM')
	# Get the first time a patient enters 4P
	first_time = FourPs_Only['IN_DTTM'].iloc[0]
	# Get the last time a patient enters 4P
	last_time = FourPs_Only['IN_DTTM'].iloc[-1]
	# Creates a list for each VisitID of the times enterted 4P
	Times_Enter_FourP = FourPs_Only.groupby('VisitID')['IN_DTTM'].apply(list)

	# Creates a dictionary whose key is each VisitID and value is the arrival time into 4P
	initial_intime_dict = dict( (key, min(Times_Enter_FourP.get(key))) for key in UniqueIDs)

	# Get list of ordered in times
	Ordered_Intimes = list(initial_intime_dict.values())

	# Generates mlist based on current season
	if month in [12,1,2]:
		mlist = [12,1,2]
	if month in [3,4,5]:
		mlist = [3,4,5]
	if month in [6,7,8]:
		mlist = [6,7,8]
	if month in [9,10,11]:
		mlist = [9,10,11]

	# Gets length of time range
	length_range = end_range - start_range

	# Create an empty dictionary to hold count of patient arrival on relevant days/time ranges
	All_Days_Dict = dict()
	# Gets the first historical day with the correct day of week
	check_time = first_time
	while check_time.weekday() != dayofweek:
		check_time += datetime.timedelta(days=1)
	# Gets the historical dates the match the season and day of week of the current date
	# Adds these dates the to All_Days_Dict dictionary
	while check_time <= last_time + datetime.timedelta(days=1):
		while (check_time.month in mlist) != True:
			check_time += datetime.timedelta(days=7)
		add_date = check_time.date()
		All_Days_Dict[add_date] = [0] * length_range
		check_time += datetime.timedelta(days=7)

	# List to hold relevant VisitIDs based on arrival time within range
	Arrival_VisitIDs = []
	# Loops through the arrival times and adds the VisitID to the list of arrival time within given range
	# Also adds increments the patient count in the dictionary for the relevant date
	for key, time in initial_intime_dict.items():
		arrival_hour = time.hour
		arrival_dayofweek = time.weekday()
		arrival_month = time.month
		if (arrival_hour >= start_range and arrival_hour < end_range) and (arrival_dayofweek == dayofweek) and (arrival_month in mlist):
			Arrival_VisitIDs.append(key)
			All_Days_Dict[time.date()][arrival_hour-start_range] += 1

	Segmented_VisitIDs = Arrival_VisitIDs

	# Obtains ADTs for patients who match temporal criteria
	Return_ADTs = ADTs[ ADTs['VisitID'].isin(Segmented_VisitIDs) ]

	# Gets the max number of patients introduced in a single hour
	max_pats = 0
	for key, val in All_Days_Dict.items():
		if max(val) > max_pats:
			max_pats = max(val)

	# Create variables used to create a pdf of number of patients introduced in a hour
	max_var = max_pats+1
	pat_probs = np.zeros(max_pats+1)
	for key, val in All_Days_Dict.items():
		for each in val:
			pat_probs[each] += 1
	pat_probs = pat_probs / (len(All_Days_Dict)*length_range)

	# Chooses the number of patients from the pdf of pat_probs
	patient_number = np.random.choice(a = max_var, p=pat_probs)

	return Return_ADTs, patient_number

''' This function segments ADTs according to temporal attributes like season, day of week, and time of arrival.
	It also chooses the number of patients to be simulated probabilistically based on historical data.
    Inputs: 
    dataframe_in- pandas dataframe of all ADTs
    Season- the season to be segmented on
    Dayofweek- the day of week to be segmented on
    start_range- the beginning of the range to segment the time of arrival
    end_ range- the end of hte range to segment time of arrival
    Outputs:
    Return_ADTs- a dataframe of ADT information only for patients who meet temporal segmenting criteria
    pat_probs- the probability distribution for the number of patients per hour entering the unit for the given time
'''
def segment_byparameters(dataframe_in,Season,Dayofweek,start_range,end_range):
	ADTs = dataframe_in

	# Obtains a list of the visitIDs from the ADT dataframe
	UniqueIDs = ADTs.VisitID.unique()

	# Generates mlist based on current season
	if Season == 0:
		mlist = [12,1,2]
	if Season == 1:
		mlist = [3,4,5]
	if Season == 2:
		mlist = [6,7,8]
	if Season == 3:
		mlist = [9,10,11]

	dayofweek = Dayofweek

	# Creates a dataframe of ADT information whose department is 4P
	FourPs_Only = ADTs[ ADTs['ADT_DEPARTMENT_NAME'].isin(['HCGH 4P ACUTE']) ]
	# Sorts the 4P dataframe by time
	FourPs_Only = FourPs_Only.sort_values('IN_DTTM')
	# Get the first time a patient enters 4P
	first_time = FourPs_Only['IN_DTTM'].iloc[0]
	# Get the last time a patient enters 4P
	last_time = FourPs_Only['IN_DTTM'].iloc[-1]
	# Creates a list for each VisitID of the times enterted 4P
	Times_Enter_FourP = FourPs_Only.groupby('VisitID')['IN_DTTM'].apply(list)

	# Creates a dictionary whose key is each VisitID and value is the arrival time into 4P
	initial_intime_dict = dict( (key, min(Times_Enter_FourP.get(key))) for key in UniqueIDs)

	# Get list of ordered in times
	Ordered_Intimes = list(initial_intime_dict.values())

	# Gets length of time range
	length_range = end_range - start_range

	# Create an empty dictionary to hold count of patient arrival on relevant days/time ranges
	All_Days_Dict = dict()
	# Gets the first historical day with the correct day of week
	check_time = first_time
	while check_time.weekday() != dayofweek:
		check_time += datetime.timedelta(days=1)
	# Gets the historical dates the match the season and day of week of the current date
	# Adds these dates the to All_Days_Dict dictionary
	while check_time <= last_time + datetime.timedelta(days=1):
		while (check_time.month in mlist) != True:
			check_time += datetime.timedelta(days=7)
		add_date = check_time.date()
		All_Days_Dict[add_date] = [0] * length_range
		check_time += datetime.timedelta(days=7)

	# List to hold relevant VisitIDs based on arrival time within range
	Arrival_VisitIDs = []
	# Loops through the arrival times and adds the VisitID to the list of arrival time within given range
	# Also adds increments the patient count in the dictionary for the relevant date
	for key, time in initial_intime_dict.items():
		arrival_hour = time.hour
		arrival_dayofweek = time.weekday()
		arrival_month = time.month
		if (arrival_hour >= start_range and arrival_hour < end_range) and (arrival_dayofweek == dayofweek) and (arrival_month in mlist):
			Arrival_VisitIDs.append(key)
			All_Days_Dict[time.date()][arrival_hour-start_range] += 1

	Segmented_VisitIDs = Arrival_VisitIDs

	# Obtains ADTs for patients who match temporal criteria
	Return_ADTs = ADTs[ ADTs['VisitID'].isin(Segmented_VisitIDs) ]

	# Gets the max number of patients introduced in a single hour
	max_pats = 0
	for key, val in All_Days_Dict.items():
		if max(val) > max_pats:
			max_pats = max(val)

	# Create variables used to create a pdf of number of patients introduced in a hour
	max_var = max_pats+1
	pat_probs = np.zeros(max_pats+1)
	for key, val in All_Days_Dict.items():
		for each in val:
			pat_probs[each] += 1
	pat_probs = pat_probs / (len(All_Days_Dict)*length_range)

	return Return_ADTs, pat_probs

